import dicomParser from "dicom-parser";
import JSZip from "jszip";
import moment from "moment";

const parseDicomFile = (file: File) => {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onload = (event) => {
      const arrayBuffer = event?.target?.result;
      try {
        const dataSet = dicomParser.parseDicom(
          new Uint8Array(arrayBuffer as any),
        );
        const desiredTags = {
          x00100020: "PatientID",
          x00100010: "PatientName",
          x00100030: "PatientBirthDate",
          x00100040: "PatientSex",
          x00380300: "PatientHistory",
          x00104000: "PatientComments",
          x001021b0: "AdditionalPatientHistory",
          x00080060: "Modality",
          x00081030: "StudyDescription",
          x00080020: "StudyDate",
          x00080030: "StudyTime",
          x00120010: "PatientMedicalHistory",
          x0020000d: "StudyInstanceUID",
          x00080050: "AccessionNumber",
          x00080080: "InstitutionName",
          x00200010: "StudyID",
          x00080090: "ReferringPhysicianName",
        };

        const patientDetails = {};

        // for (const prop in desiredTags) {
        //   const tagValue = dataSet.string(prop);
        //   if (tagValue) {
        //     patientDetails[desiredTags[prop]] = tagValue;
        //   }
        // }

        resolve(patientDetails);
      } catch (error) {
        reject(error);
      }
    };
    reader.onerror = (error) => reject(error);
    reader.readAsArrayBuffer(file);
  });
};

export const handleZipFile = async (file: File) => {
  const jszip = new JSZip();
  const zip = await jszip.loadAsync(file);
  const dicomFiles = [];

  for (const filename of Object.keys(zip.files)) {
    try {
      const fileData = await zip.files[filename].async("arraybuffer");

      const dataSet = dicomParser.parseDicom(new Uint8Array(fileData));

      // Extract desired tags from the DICOM data
      const desiredTags = {
        x00100020: "PatientID",
        x00100010: "PatientName",
        x00100030: "PatientBirthDate",
        x00100040: "PatientSex",
        x00380300: "PatientHistory",
        x00104000: "PatientComments",
        x001021b0: "AdditionalPatientHistory",
        x00080060: "Modality",
        x00081030: "StudyDescription",
        x00080020: "StudyDate",
        x00080030: "StudyTime",
        x00120010: "PatientMedicalHistory",
        x0020000d: "StudyInstanceUID",
        x00080050: "AccessionNumber",
        x00080080: "InstitutionName",
        x00200010: "StudyID",
        x00080090: "ReferringPhysicianName",
      };

      const patientDetails = {};

      // for (const prop in desiredTags) {
      //   const tagValue = dataSet.string(prop);
      //   if (tagValue) {
      //     patientDetails[desiredTags[prop]] = tagValue;
      //   }
      // }
      dicomFiles.push(patientDetails);
      // dicomFiles.push(dataSet);
    } catch (error) {
      console.log(error);
    }
  }
  return dicomFiles;
};

export const getDataSet = async (file: File) => {
  if (file.name.endsWith(".zip")) {
    return await handleZipFile(file);
  } else if (file.name.endsWith(".dcm")) {
    return [await parseDicomFile(file)];
  } else {
    throw new Error("Unsupported file type");
  }
};

export const getDistinctStudiesFromDataset = (studies: any) => {
  const uniqueStudies = [];
  const seen = new Set();

  for (const study of studies) {
    if (study.StudyInstanceUID && !seen.has(study.StudyInstanceUID)) {
      uniqueStudies.push(study);
      seen.add(study.StudyInstanceUID);
    }
  }
  return uniqueStudies;
};

export const formattedName = (name: string) => {
  const format = name
    .split(" ")
    .map((word) => word.trim())
    .join("");
  return format;
};

export function getFileExtension(filename: string) {
  return filename
    .slice(((filename.lastIndexOf(".") - 1) >>> 0) + 2)
    .toLowerCase();
}

export const readFileAsBinary = (file: File) => {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.readAsArrayBuffer(file);
    reader.onload = () => resolve(reader.result);

    reader.onerror = (error) => reject(error);
  });
};

export function readChunkAsString(file: File): Promise<string> {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();

    reader.onload = () => {
      if (reader.result) {
        const result =
          typeof reader.result === "string"
            ? reader.result.split(",")[1]
            : null;
        resolve(result || "");
      } else {
        reject(new Error("Failed to read file"));
      }
    };

    reader.onerror = () => reject(new Error("Error reading file"));
    reader.readAsDataURL(file);
  });
}

export function getDistinctParentPatients(data: any) {
  const distinctParentPatients = new Set();
  const result = [];

  for (const item of data) {
    if (!distinctParentPatients.has(item.ParentPatient)) {
      distinctParentPatients.add(item.ParentPatient);
      result.push(item);
    }
  }

  return result;
}

export function getDistinctParentStudy(data: any) {
  const distinctParentStudies = new Set();
  const result = [];

  for (const item of data) {
    if (!distinctParentStudies.has(item.ParentStudy)) {
      distinctParentStudies.add(item.ParentStudy);
      result.push(item);
    }
  }

  return result;
}

export const setGender = (gender: string) => {
  if (gender) {
    if (
      gender.toLowerCase() == "m" ||
      gender.toLowerCase() == "f" ||
      gender.toLowerCase() == "o"
    ) {
      const gendr = {
        M: "male",
        F: "female",
        O: "other",
      };

      // return gendr[gender];
    } else {
      return gender;
    }
  }
  return "";
};

export const convertDateFormat = (studyDate: string) => {
  let formattedDate;

  if (/^\d{8}$/.test(studyDate)) {
    formattedDate = moment(studyDate, "YYYYMMDD").toISOString();
  } else {
    formattedDate = moment(studyDate).toISOString();
  }

  return formattedDate;
};
