import { UploadSlideData } from "@/interfaces";
import env from "../../env";
import { request } from "./api";

const { API_URL } = env;

const URL = {
  institution: `${API_URL}/institutions`,
  pathologist: `${API_URL}/pathologists`,
};

function uploadSlideInstitution({
  institutionId,
  body,
}: {
  institutionId: number;
  body: UploadSlideData;
}) {
  return request(
    `${URL.institution}/${institutionId}/slide-images`,
    body,
    "POST",
  );
}

function getInstitutionSlides({
  institutionId,
  search,
  searchFields,
}: {
  institutionId: number;
  search: string;
  searchFields: string[];
}) {
  return request(
    `${URL.institution}/${institutionId}/slide-images`,
    { search: search, searchFields: searchFields.join(",") },
    "GET",
  );
}
function getInstitutionSharedSlides({
  institutionId,
  search,
  searchFields,
}: {
  institutionId: number;
  search: string;
  searchFields: string[];
}) {
  return request(
    `${URL.institution}/${institutionId}/slide-images/shared`,
    { search: search, searchFields: searchFields.join(",") },
    "GET",
  );
}

function getInstitutionSlideDetail({
  institutionId,
  slideId,
}: {
  institutionId: number;
  slideId: string;
}) {
  return request(
    `${URL.institution}/${institutionId}/slide-images/${slideId}`,
    {},
    "GET",
  );
}

// Report

function createInstitutionSlideReport({
  institutionId,
  slideId,
  body,
}: {
  institutionId: number;
  slideId: string;
  body: FormData;
}) {
  return request(
    `${URL.institution}/${institutionId}/slide-images/${slideId}/reports`,
    body,
    "POST",
    {},
    { isFormData: true },
  );
}

function shareReportWithConsultant({
  institutionId,
  reportId,
  pathologistId,
}: {
  institutionId: number;
  reportId: number;
  pathologistId: number;
}) {
  return request(
    `${URL.institution}/${institutionId}/report/${reportId}/share/consultant`,
    {
      pathologistId,
    },
    "PATCH",
  );
}
function shareReportExternal({
  institutionId,
  reportId,
  externalShareInfo,
}: {
  institutionId: number;
  reportId: number;
  externalShareInfo: {
    name: string;
    email: string;
    phoneNumber: string;
    validityPeriod: number;
    code: string;
  };
}) {
  return request(
    `${URL.institution}/${institutionId}/report/${reportId}/share/external
`,

    externalShareInfo,
    "PATCH",
  );
}

function addCommentToInstitutionReport({
  institutionId,
  reportId,
  comment,
}: {
  institutionId: number;
  reportId: number;
  comment: string;
}) {
  return request(
    `${URL.institution}/${institutionId}/reports/${reportId}/second-opinion`,
    { text: comment },
    "POST",
  );
}

function getCommentsForInstitutionReport({
  institutionId,
  reportId,
}: {
  institutionId: number;
  reportId: number;
}) {
  return request(
    `${URL.institution}/${institutionId}/reports/${reportId}/comments`,
    {},
    "GET",
  );
}

function approveInstitutionReport({
  institutionId,
  reportId,
}: {
  institutionId: number;
  reportId: number;
}) {
  return request(
    `${URL.institution}/${institutionId}/reports/${reportId}/approve`,
    {},
    "POST",
  );
}

function externalOpinion({
  shareId,
  body,
}: {
  shareId: number;
  body: {
    email: string;
    code: string;
    text: string;
    reportId: number;
  };
}) {
  return request(`${URL.institution}/${shareId}/second-opinion`, body, "POST");
}

function verifySharedReport({
  slideId,
  email,
  code,
}: {
  slideId: string;
  email: string;
  code: string;
}) {
  return request(
    `${URL.institution}/${slideId}/verify`,
    { email, code },
    "POST",
  );
}

// Pathologist

function getPathologistSlides({
  pathologistId,
  search,
  searchFields,
}: {
  pathologistId: number;
  search: string;
  searchFields: string[];
}) {
  return request(
    `${URL.pathologist}/${pathologistId}/slide-images`,
    { search: search, searchFields: searchFields.join(",") },
    "GET",
  );
}

function getPathologistSlideDetail({
  pathologistId,
  slideId,
}: {
  pathologistId: number;
  slideId: string;
}) {
  return request(
    `${URL.pathologist}/${pathologistId}/slide-images/${slideId}`,
    {},
    "GET",
  );
}

function uploadSlidePersonal({
  pathologistId,
  body,
}: {
  pathologistId: number;
  body: UploadSlideData;
}) {
  return request(
    `${URL.pathologist}/${pathologistId}/slide-images`,
    body,
    "POST",
  );
}

function createPathologistSlideReport({
  pathologistId,
  slideId,
  body,
}: {
  pathologistId: number;
  slideId: string;
  body: FormData;
}) {
  return request(
    `${URL.pathologist}/${pathologistId}/slide-images/${slideId}/reports`,
    body,
    "POST",
    {},
    { isFormData: true },
  );
}

const slideService = {
  uploadSlideInstitution,
  getInstitutionSlides,
  getInstitutionSlideDetail,
  createInstitutionSlideReport,
  addCommentToInstitutionReport,
  getCommentsForInstitutionReport,
  approveInstitutionReport,
  shareReportWithConsultant,
  shareReportExternal,
  externalOpinion,
  verifySharedReport,
  getInstitutionSharedSlides,
  getPathologistSlides,
  getPathologistSlideDetail,
  uploadSlidePersonal,
  createPathologistSlideReport,
};

export default slideService;
