import { modalityTypes } from "@/interfaces";
import env from "../../env";
import { request } from "./api";

const { API_URL } = env;

const URL = {
  pathologist: `${API_URL}/pathologists`,
};

function getPathologistDetails({ pathologistId }: { pathologistId: number }) {
  return request(`${URL.pathologist}/${pathologistId}/me`, {}, "GET");
}

function getPathologistDashboard({ pathologistId }: { pathologistId: number }) {
  return request(
    `${URL.pathologist}/${pathologistId}/dashboard-data`,
    {},
    "GET",
  );
}

function getPathologistStatistics({
  pathologistId,
  startDate,
  endDate,
}: {
  pathologistId: number;
  startDate?: string;
  endDate?: string;
}) {
  return request(
    `${URL.pathologist}/${pathologistId}/slide-images/statistics`,
    { startDate: startDate, endDate: endDate },
    "GET",
  );
}

function getPathologistDistribution({
  pathologistId,
  startDate,
  endDate,
}: {
  pathologistId: number;
  startDate?: string;
  endDate?: string;
}) {
  return request(
    `${URL.pathologist}/${pathologistId}/slide-images/distribution`,
    { startDate: startDate, endDate: endDate },
    "GET",
  );
}

function updateSlideFormData({
  pathologistId,
  slideId,
  body,
}: {
  pathologistId: number;
  slideId?: number | string;
  body: FormData;
}) {
  return request(
    `${URL.pathologist}/${pathologistId}/slide-images/${slideId}`,
    body,
    "PATCH",
    {},
    { isFormData: true },
  );
}

function updateSlideData({
  pathologistId,
  slideId,
  ...fields
}: {
  pathologistId: number;
  slideId?: number | string;
  name?: string;
  patientName?: string;
  modality?: modalityTypes | string;
  comment?: string;
  patientGender?: string;
  macroscopicDescription?: string;
  clinicalHistory?: string;
  nature?: string;
  dob?: string;
  dateOfSampleCollection?: string;
  referringPhysician?: string;
  referringFacility?: string;
}) {
  const filteredPayload = Object.fromEntries(
    Object.entries(fields).filter(
      ([_, value]) => value !== undefined && value !== null && value !== "",
    ),
  );
  return request(
    `${URL.pathologist}/${pathologistId}/slide-images/${slideId}`,
    filteredPayload,
    "PATCH",
    {},
  );
}

function createPersonalSlideReport({
  pathologistId,
  slideId,
  body,
}: {
  pathologistId: number;
  slideId: string;
  body: FormData;
}) {
  return request(
    `${URL.pathologist}/${pathologistId}/slide-images/${slideId}/reports`,
    body,
    "POST",
    {},
    { isFormData: true },
  );
}

function approvePersonalReport({
  pathologistId,
  reportId,
}: {
  pathologistId: number;
  reportId: number;
}) {
  return request(
    `${URL.pathologist}/${pathologistId}/reports/${reportId}/approve`,
    {},
    "PATCH",
  );
}

function getCommentsForPersonalReport({
  pathologistId,
  reportId,
}: {
  pathologistId: number;
  reportId: number;
}) {
  return request(
    `${URL.pathologist}/${pathologistId}/reports/${reportId}/comments`,
    {},
    "GET",
  );
}

function addCommentToPersonalReport({
  pathologistId,
  reportId,
  comment,
}: {
  pathologistId: number;
  reportId: number;
  comment: string;
}) {
  return request(
    `${URL.pathologist}/${pathologistId}/reports/${reportId}/comments`,
    { comment },
    "POST",
  );
}

const pathologistService = {
  getPathologistDetails,
  getPathologistDashboard,
  getPathologistStatistics,
  getPathologistDistribution,
  updateSlideFormData,
  updateSlideData,
  createPersonalSlideReport,
  approvePersonalReport,
  getCommentsForPersonalReport,
  addCommentToPersonalReport,
};

export default pathologistService;
