import env from "../../env";
import { request } from "./api";

import { modalityTypes } from "@/interfaces";

const { API_URL } = env;

const URL = {
  institution: `${API_URL}/institutions`,
};

function getInstitutionDetails({ institutionId }: { institutionId: number }) {
  return request(`${URL.institution}/${institutionId}`, {}, "GET");
}

function getInstitutionPathologistUserDetails({
  institutionId,
}: {
  institutionId: number;
}) {
  return request(
    `${URL.institution}/${institutionId}/pathologist/me`,
    {},
    "GET",
  );
}

function logIntoInstitution({
  institutionId,
  loginAs,
}: {
  institutionId: number;
  loginAs: "PATHOLOGIST" | "RADIOLOGIST" | "ADMIN" | "SUPERADMIN";
}) {
  return request(
    `${URL.institution}/${institutionId}/login`,
    { loginAs },
    "POST",
  );
}

function getInstitutionContacts({ institutionId }: { institutionId: number }) {
  return request(`${URL.institution}/${institutionId}/users`, {}, "GET");
}

function addPathologist({
  institutionId,
  firstName,
  lastName,
  email,
  type,
}: {
  institutionId: number;
  firstName: string;
  lastName: string;
  email: string;
  type: string;
}) {
  return request(
    `${URL.institution}/${institutionId}/pathologists`,
    { firstName, lastName, email, type },
    "POST",
  );
}

function changePathologistType({
  institutionId,
  pathologistId,
  type,
}: {
  institutionId: number;
  pathologistId: number;
  type: string;
}) {
  return request(
    `${URL.institution}/${institutionId}/pathologists/${pathologistId}`,
    { type },
    "PATCH",
  );
}

function getInstitutionPathologists({
  institutionId,
  search,
  searchFields,
}: {
  institutionId: number;
  search: string;
  searchFields: string[];
}) {
  return request(
    `${URL.institution}/${institutionId}/pathologists`,
    { search: search, searchFields: searchFields.join(",") },
    "GET",
  );
}

function removePathologist({
  institutionId,
  pathologistId,
}: {
  institutionId: number;
  pathologistId: number;
}) {
  return request(
    `${URL.institution}/${institutionId}/pathologists/${pathologistId}`,
    {},
    "DELETE",
  );
}

function getInstitutionStatistics({
  institutionId,
}: {
  institutionId: number;
}) {
  return request(
    `${URL.institution}/${institutionId}/slide-images/statistics`,
    {},
    "GET",
  );
}

function getInstitutionDistribution({
  institutionId,
}: {
  institutionId: number;
}) {
  return request(
    `${URL.institution}/${institutionId}/slide-images/distribution`,
    {},
    "GET",
  );
}

function getInstitutionDashboard({ institutionId }: { institutionId: number }) {
  return request(
    `${URL.institution}/${institutionId}/dashboard-data`,
    {},
    "GET",
  );
}

function updateSlideData({
  institutionId,
  slideId,
  ...fields
}: {
  institutionId: number;
  slideId?: number | string;
  name?: string;
  patientName?: string;
  modality?: modalityTypes | string;
  comment?: string;
  patientGender?: string;
  macroscopicDescription?: string;
  clinicalHistory?: string;
  nature?: string;
  dob?: string;
  dateOfSampleCollection?: string;
  referringPhysician?: string;
  referringFacility?: string;
}) {
  // Filter out empty strings, null, and undefined
  const filteredPayload = Object.fromEntries(
    Object.entries(fields).filter(
      ([_, value]) => value !== undefined && value !== null && value !== "",
    ),
  );

  return request(
    `${URL.institution}/${institutionId}/slide-images/${slideId}`,
    filteredPayload,
    "PATCH",
  );
}

function updateSlideDataFormData({
  institutionId,
  slideId,
  body,
}: {
  institutionId: number;
  slideId?: number | string;
  body: FormData;
}) {
  return request(
    `${URL.institution}/${institutionId}/slide-images/${slideId}`,
    body,
    "PATCH",
    {},
    { isFormData: true },
  );
}

function uploadLetterHead({
  institutionId,
  body,
}: {
  institutionId: number;
  body: FormData;
}) {
  return request(
    `${URL.institution}/${institutionId}/letterhead`,
    body,
    "PATCH",
    {},
    { isFormData: true },
  );
}

function removeLetterHead({ institutionId }: { institutionId: number }) {
  return request(
    `${URL.institution}/${institutionId}/letterhead`,
    {},
    "DELETE",
  );
}

function uploadLogo({
  institutionId,
  body,
}: {
  institutionId: number;
  body: FormData;
}) {
  return request(
    `${URL.institution}/${institutionId}/logo`,
    body,
    "PATCH",
    {},
    { isFormData: true },
  );
}

function removeLogo({ institutionId }: { institutionId: number }) {
  return request(`${URL.institution}/${institutionId}/logo`, {}, "DELETE");
}

const institutionService = {
  getInstitutionDetails,
  getInstitutionPathologistUserDetails,
  logIntoInstitution,
  getInstitutionContacts,
  addPathologist,
  getInstitutionPathologists,
  getInstitutionDashboard,
  getInstitutionDistribution,
  getInstitutionStatistics,
  updateSlideData,
  removePathologist,
  changePathologistType,
  updateSlideDataFormData,
  uploadLetterHead,
  removeLetterHead,
  uploadLogo,
  removeLogo,
};

export default institutionService;
