import { getToken } from "@/utils/sessions/session.utils";
import axios, { AxiosResponse } from "axios";

interface RequestOptions {
  isFormData?: boolean;
  isPdf?: boolean;
  isCsv?: boolean;
  isDownload?: boolean;
  fileName?: string;
}

export interface ApiResponse<T = unknown> {
  data?: T;
  message: string;
  success: boolean;
  [key: string]: unknown;
}

export const defaultHeader: Record<string, string> = {
  Accept: "application/json",
  "Cache-Control": "no-cache",
  "Content-Type": "application/json",
};

export async function request<T, P = unknown>(
  url: string,
  payload: P,
  method: "GET" | "POST" | "PATCH" | "DELETE",
  header?: Record<string, string>,
  options: RequestOptions = {},
): Promise<ApiResponse<T>> {
  delete defaultHeader.Authorization;
  if (options?.isFormData) delete defaultHeader["Content-Type"];
  if (options?.isPdf) defaultHeader["Content-Type"] = "application/pdf";
  if (options?.isCsv) defaultHeader["Content-Type"] = "text/csv";

  const requestMethod = method.toLowerCase() as
    | "get"
    | "post"
    | "patch"
    | "delete";

  const persistedToken = getToken();

  if (persistedToken) {
    defaultHeader.Authorization = `Bearer ${persistedToken}`;
  }

  try {
    const response: AxiosResponse<ApiResponse<T>> = await axios.request({
      url,
      method: requestMethod,
      headers: { ...defaultHeader, ...header },
      responseType: !options.isDownload && !options.isPdf ? "json" : "blob",
      [requestMethod === "get" ? "params" : "data"]: payload,
    });

    const jsonResponse = response.data;

    if (!options.isDownload && !jsonResponse.arrayBuffer) {
      return jsonResponse;
    }

    if (options.isDownload) {
      const objectUrl = window.URL.createObjectURL(
        jsonResponse as unknown as Blob,
      );
      const a = document.createElement("a");
      a.href = objectUrl;
      a.download = options.fileName || "file";
      document.body.appendChild(a);
      a.click();
      a.remove();

      return jsonResponse;
    } else {
      return {
        success: jsonResponse.success,
        message: jsonResponse.message,
        data: window.URL.createObjectURL(jsonResponse as unknown as Blob) as T,
      };
    }
  } catch (error: unknown) {
    console.log(error);
    if (axios.isAxiosError(error)) {
      throw {
        success: false,
        message:
          error.response?.data?.message ||
          error.response?.data?.error ||
          "An error occurred",
        errors: error.response?.data?.errors || "An error occurred",
      };
    }
    throw {
      success: false,
      message: "An unknown error occurred",
      data: null,
    };
  }
}
