import { createSlice, PayloadAction } from "@reduxjs/toolkit";

interface GlobalState {
  drawer: {
    isOpen: boolean;
    closable: boolean;
    width: string;
  };
  drawerChildren: { key: string | null };
  firstPopup: {
    isOpen: boolean;
    closable: boolean;
  };
  secondPopup: {
    isOpen: boolean;
    closable: boolean;
  };
  popupChildren: {
    firstPopup: {
      componentKey: string | null;
      props: Record<string, any>;
    };
    secondPopup: {
      componentKey: string | null;
      props: Record<string, any>;
    };
  };
  width: number | null;
  showAlert: boolean;
  alertChildren: any;
  modalLoading: boolean;
  sideBarClosed: boolean;
  promptLoading: boolean;
}

// Initial state
const initialState: GlobalState = {
  drawer: {
    isOpen: false,
    closable: true,
    width: "",
  },
  drawerChildren: { key: null },
  firstPopup: {
    isOpen: false,
    closable: false,
  },
  secondPopup: {
    isOpen: false,
    closable: false,
  },
  popupChildren: {
    firstPopup: { componentKey: null, props: {} },
    secondPopup: { componentKey: null, props: {} },
  },
  width: null,
  showAlert: false,
  alertChildren: null,
  modalLoading: false,
  sideBarClosed: true,
  promptLoading: false,
};

// Create the slice
const globalSlice = createSlice({
  name: "global",
  initialState,
  reducers: {
    setDrawer: (
      state,
      action: PayloadAction<{
        isOpen: boolean;
        closable?: boolean;
        width?: string;
      }>,
    ) => {
      state.drawer = {
        isOpen: action.payload.isOpen,
        closable: action.payload.closable ?? true,
        width: action.payload.width || "",
      };
    },
    setDrawerChildren: (
      state,
      action: PayloadAction<{ key: string | null }>,
    ) => {
      state.drawerChildren = {
        key: action.payload.key,
      };
    },

    togglePopup: (
      state,
      action: PayloadAction<{
        key: "firstPopup" | "secondPopup";
        isOpen: boolean;
        closable?: boolean;
      }>,
    ) => {
      state[action.payload.key] = {
        ...state[action.payload.key],
        isOpen: action.payload.isOpen,
        closable: action?.payload?.closable || true,
      };
    },
    setPopupChildren: (
      state,
      action: PayloadAction<{
        key: "firstPopup" | "secondPopup";
        componentKey: string;
        props?: Record<string, any>;
      }>,
    ) => {
      const { key, componentKey, props } = action.payload;
      if (state.popupChildren[key]) {
        state.popupChildren[key].componentKey = componentKey;
        state.popupChildren[key].props = props || {};
      }
    },
    setModalLoading: (state, action: PayloadAction<boolean>) => {
      state.modalLoading = action.payload;
    },
    setSideBarClosed: (state, action: PayloadAction<boolean>) => {
      state.sideBarClosed = action.payload;
    },
    setPromptLoading: (state, action: PayloadAction<boolean>) => {
      state.promptLoading = action.payload;
    },
  },
});

// Export actions
export const {
  setDrawer,
  setDrawerChildren,
  togglePopup,
  setPopupChildren,
  setModalLoading,
  setSideBarClosed,
  setPromptLoading,
} = globalSlice.actions;

// Export reducer
export default globalSlice.reducer;
