// redux/call/callSlice.ts
import { createSlice, PayloadAction } from "@reduxjs/toolkit";

interface UserOnCall {
  id: string;
  stream?: MediaStream;
  isMuted?: boolean;
  firstName?: string;
  lastName?: string;
  peerId?: string;
}

interface IncomingCallPayload {
  callerId: string;
  callerName: string;
  callType: "voice" | "video";
  meetingId: string;
}

interface CallState {
  hasLocalStream: boolean;
  remoteStream: MediaStream | null;
  activeStream: MediaStream | null;
  onCall: boolean;
  outgoingCall: string | null; // meetingId of outgoing call
  incomingCall: IncomingCallPayload | null;
  caller: boolean | string | null; // ID of the caller
  callType: "voice" | "video" | null;
  usersOnCall: UserOnCall[];
  currentMeetingId: string | null;
}

const initialState: CallState = {
  hasLocalStream: false,
  remoteStream: null,
  activeStream: null,
  onCall: false,
  outgoingCall: null,
  incomingCall: null,
  caller: null,
  callType: null,
  usersOnCall: [],
  currentMeetingId: null,
};

const callSlice = createSlice({
  name: "call",
  initialState,
  reducers: {
    setLocalStream: (state, action: PayloadAction<boolean>) => {
      state.hasLocalStream = action.payload;
    },
    setRemoteStream: (state, action: PayloadAction<MediaStream | null>) => {
      state.remoteStream = action.payload;
    },
    setActiveStream: (state, action: PayloadAction<MediaStream | null>) => {
      state.activeStream = action.payload;
    },
    setOnCall: (state, action: PayloadAction<boolean>) => {
      state.onCall = action.payload;
    },
    setOutgoingCall: (state, action: PayloadAction<string | null>) => {
      state.outgoingCall = action.payload;
    },
    setIncomingCall: (
      state,
      action: PayloadAction<IncomingCallPayload | null>,
    ) => {
      state.incomingCall = action.payload;
      state.caller = action.payload?.callerId || null;
    },
    setCaller: (state, action: PayloadAction<boolean | string | null>) => {
      state.caller = action.payload;
    },
    setCallType: (state, action: PayloadAction<"voice" | "video" | null>) => {
      state.callType = action.payload;
    },
    setUsersOnCall: (state, action: PayloadAction<UserOnCall[]>) => {
      state.usersOnCall = action.payload;
    },
    addUserOnCall: (state, action: PayloadAction<UserOnCall>) => {
      if (!state.usersOnCall.some((user) => user.id === action.payload.id)) {
        state.usersOnCall.push(action.payload);
      }
    },
    removeUserFromCall: (state, action: PayloadAction<string>) => {
      state.usersOnCall = state.usersOnCall.filter(
        (user) => user.id !== action.payload,
      );
    },
    updateUserOnCall: (
      state,
      action: PayloadAction<Partial<UserOnCall> & { id: string }>,
    ) => {
      const index = state.usersOnCall.findIndex(
        (user) => user.id === action.payload.id,
      );
      if (index !== -1) {
        state.usersOnCall[index] = {
          ...state.usersOnCall[index],
          ...action.payload,
        };
      }
    },
    setCurrentMeetingId: (state, action: PayloadAction<string | null>) => {
      state.currentMeetingId = action.payload;
    },
    resetCallState: () => initialState,
  },
});

export const {
  setLocalStream,
  setRemoteStream,
  setActiveStream,
  setOnCall,
  setOutgoingCall,
  setIncomingCall,
  setCaller,
  setCallType,
  setUsersOnCall,
  addUserOnCall,
  removeUserFromCall,
  updateUserOnCall,
  setCurrentMeetingId,
  resetCallState,
} = callSlice.actions;

// export type CallStateType = ReturnType<typeof callSlice.reducer>;
export default callSlice.reducer;
