import { setSideBarClosed } from "@/redux/global/globalSlice";
import { AppDispatch } from "@/redux/store";
import { getFileExtFromBase64 } from "@/utils/utilities";
import React, { useRef, useState } from "react";
import toast from "react-hot-toast";
import { useDispatch } from "react-redux";
import { useScreenshot } from "use-react-screenshot";

const useScreenTools = () => {
  const mediaRecorderRef = useRef<MediaRecorder | null>(null);
  const mediaElementRef = useRef<HTMLVideoElement | null>(null);

  const [image, takeScreenshot] = useScreenshot();
  const [dataStream, setDataStream] = useState<Blob[]>([]);
  const [recording, setRecording] = useState(false);
  const [videoFile, setVideoFile] = useState<Blob | null>(null);
  const [imageFile, setImageFile] = useState(image);
  const [isCaptured, setIsCaptured] = useState(false);

  const reduxDispatch = useDispatch<AppDispatch>();

  const onStartScreenRecording = async () => {
    const audioStream = await navigator.mediaDevices.getUserMedia({
      audio: true,
    });
    navigator.mediaDevices
      .getDisplayMedia({
        video: true,
        audio: true,
      })
      .then((stream) => {
        const chunks: BlobPart[] = [];

        let mediaStream;
        if (!audioStream) {
          mediaStream = stream;
        } else {
          mediaStream = new MediaStream([
            ...stream.getVideoTracks(),
            ...audioStream.getAudioTracks(),
          ]);
        }

        const mediaRecorder = new MediaRecorder(mediaStream);
        mediaRecorderRef.current = mediaRecorder;

        mediaRecorder.ondataavailable = (e) => {
          chunks.push(e.data);
          setDataStream((prev) => [...prev, e.data]);
        };

        mediaRecorder.start();
        mediaRecorder.onstart = () => {
          setRecording(true);
        };

        mediaRecorder.onstop = () => {
          setRecording(false);
          mediaStream.getTracks().forEach((track) => track.stop());
          const blob = new Blob(chunks, { type: "video/webm" });
          setVideoFile(blob);
          setIsCaptured(true);
        };
      });
  };

  const handleStopRecording = () => {
    mediaRecorderRef.current?.stop();
    setIsCaptured(true);
  };

  const downloadScreenRecording = () => {
    const time = new Date().toISOString();
    const fileName = `xolani-screen-recording-${time}.webm`;

    if (!videoFile) {
      console.error("No screen recording available to download.");
      return;
    }

    const link = document.createElement("a");
    link.href = URL.createObjectURL(videoFile);
    link.download = fileName;
    link.click();

    setVideoFile(null);
    setIsCaptured(false);
  };

  const handleScreenshot = () => {
    const appLayout = document.querySelector("#app-layout") as HTMLElement; // Type assertion

    if (!appLayout) {
      toast.error("App layout not found!");
      return;
    }

    // Create a div element for dimming overlay
    const dimmedOverlay = document.createElement("div");
    dimmedOverlay.style.position = "fixed";
    dimmedOverlay.style.top = "0";
    dimmedOverlay.style.left = "0";
    dimmedOverlay.style.width = "100%";
    dimmedOverlay.style.height = "100%";
    dimmedOverlay.style.backgroundColor = "rgba(0, 0, 0, 0.5)";
    dimmedOverlay.style.zIndex = "9999";
    document.body.appendChild(dimmedOverlay);

    dimmedOverlay.animate(
      [
        { transform: "scale(0)", opacity: "0" },
        { transform: "scale(1)", opacity: "0.5" },
      ],
      {
        duration: 300,
        easing: "ease-in-out",
        fill: "forwards",
      },
    ).onfinish = () => {
      takeScreenshot(appLayout)
        .then((result) => {
          setImageFile(result);
          toast.success("Screen shot captured!");
          setIsCaptured(true);
          // Zoom-out animation
          dimmedOverlay.animate(
            [
              { transform: "scale(1)", opacity: "0.5" },
              { transform: "scale(0)", opacity: "0" },
            ],
            {
              duration: 300,
              easing: "ease-in-out",
              fill: "forwards",
            },
          ).onfinish = () => {
            document.body.removeChild(dimmedOverlay); // Remove the overlay after animation completes
          };
        })
        .catch((error) => {
          toast.error(error?.message);
          document.body.removeChild(dimmedOverlay); // Remove the overlay in case of error
        })
        .finally(() => reduxDispatch(setSideBarClosed(false)));
    };
  };

  const downloadScreenshot = () => {
    if (imageFile) {
      const fileExt = getFileExtFromBase64(imageFile);
      const time = new Date().toISOString();
      const fileName = `xolani-screenshot-${time}.${fileExt}`;
      const link = document.createElement("a");
      link.href = imageFile;
      link.download = fileName;
      link.click();
      setImageFile(null);
      setIsCaptured(false);
    }
  };

  const removeFile = () => {
    setVideoFile(null);
    setImageFile(null);
    setIsCaptured(false);
  };

  return {
    videoFile,
    imageFile,
    onStartScreenRecording,
    handleStopRecording,
    downloadScreenRecording,
    recording,
    isCaptured,
    mediaElementRef,
    removeFile,
    downloadScreenshot,
    handleScreenshot,
  };
};

export default useScreenTools;
