import { useReducer, useRef, useState, useCallback, useEffect } from "react";
import { useDispatch, useSelector } from "react-redux";
import { RootState, AppDispatch } from "@/redux/store";
import toast from "react-hot-toast";
import {
  setCaller,
  setCallType,
  setCurrentMeetingId,
  setIncomingCall,
  setLocalStream,
  setOnCall,
  setOutgoingCall,
} from "@/redux/call/callSlice";
import { generateMeetingId } from "@/utils/utilities";

declare global {
  interface MediaTrackConstraints {
    cursor?: ConstrainDOMString;
    displaySurface?: ConstrainDOMString;
  }
}

interface CallState {
  audioMode: boolean;
  videoMode: boolean;
  screenMode: boolean;
  fullScreen: boolean;
  microphonePermission: PermissionState;
  cameraPermission: PermissionState;
}

interface UserOnCall {
  id: string;
  peerCall: {
    peerConnection: RTCPeerConnection;
    peerId: string;
  };
  stream?: MediaStream;
  isMuted?: boolean;
  firstName?: string;
  lastName?: string;
}

// interface MediaTracks {
//   audio: MediaStreamTrack | null;
//   video: MediaStreamTrack | null;
//   screen: MediaStreamTrack[];
// }

const initialState: CallState = {
  audioMode: false,
  videoMode: false,
  screenMode: false,
  fullScreen: false,
  microphonePermission: "prompt",
  cameraPermission: "prompt",
};
interface UseCallProps {
  localStreamRef?: React.RefObject<HTMLVideoElement>;
}

const useCall = ({ localStreamRef }: UseCallProps = {}) => {
  const localStream = useRef<MediaStream | null>(null);
  const remoteStreamRef = useRef<HTMLVideoElement>(null);
  // const mediaTracks = useRef<MediaTracks>({
  //   audio: null,
  //   video: null,
  //   screen: [],
  // });
  const myAudioTrack = useRef<MediaStreamTrack | null>(null);
  const myVideoTrack = useRef<MediaStreamTrack | null>(null);
  const myScreenTrack = useRef<MediaStreamTrack[]>([]);

  const { onCall, callType } = useSelector((state: RootState) => state.call);
  // const { activeUser } = useSelector((state: RootState) => state.user);

  const [state, dispatch] = useReducer(
    (prevState: CallState, update: Partial<CallState>) => ({
      ...prevState,
      ...update,
    }),
    initialState,
  );

  const reduxDispatch = useDispatch<AppDispatch>();
  const [usersOnCall, setUsersOnCall] = useState<UserOnCall[]>([]);
  const [isMediaInitialized, setIsMediaInitialized] = useState(false);

  // Permission handling
  useEffect(() => {
    const handlePermissionChange =
      (type: "microphone" | "camera") => (event: Event) => {
        const status = event.target as PermissionStatus;
        dispatch({
          [type === "microphone" ? "microphonePermission" : "cameraPermission"]:
            status.state,
        });
      };

    const queryPermissions = async () => {
      try {
        const [micPermission, camPermission] = await Promise.all([
          navigator.permissions.query({ name: "microphone" }),
          navigator.permissions.query({ name: "camera" }),
        ]);

        dispatch({
          microphonePermission: micPermission.state,
          cameraPermission: camPermission.state,
        });

        micPermission.onchange = handlePermissionChange("microphone");
        camPermission.onchange = handlePermissionChange("camera");
      } catch (error) {
        console.error("Permission query error:", error);
      }
    };

    queryPermissions();

    return () => {
      navigator.permissions.query({ name: "microphone" }).then((permission) => {
        permission.onchange = null;
      });
      navigator.permissions.query({ name: "camera" }).then((permission) => {
        permission.onchange = null;
      });
    };
  }, []);

  // Media device management
  const getDeviceStream = useCallback(
    async (type: "voice" | "video" = "voice") => {
      try {
        const stream = await navigator.mediaDevices.getUserMedia({
          video: type === "video",
          audio: true,
        });

        // Clear previous tracks
        myAudioTrack.current?.stop();
        myVideoTrack.current?.stop();

        // Update media tracks
        myAudioTrack.current = stream.getAudioTracks()[0] || null;
        myVideoTrack.current = stream.getVideoTracks()[0] || null;

        setIsMediaInitialized((prev) => !prev);

        localStream.current = stream;
        reduxDispatch(setLocalStream(true));

        // Handle video element
        if (localStreamRef?.current) {
          localStreamRef.current.srcObject = stream;
        }

        // Store tracks and update state
        stream.getTracks().forEach((track) => {
          if (track.kind === "audio") {
            myAudioTrack.current = track;
            dispatch({
              audioMode: track.enabled,
              microphonePermission: "granted",
            });
          }
          if (track.kind === "video") {
            myVideoTrack.current = track;
            dispatch({
              videoMode: type === "video",
              cameraPermission: "granted",
            });
          }
        });
      } catch (error) {
        const err = error as DOMException;
        toast.error(`Device access error: ${err.message}`);
        dispatch({
          microphonePermission:
            err.name === "NotAllowedError" ? "denied" : "prompt",
          cameraPermission:
            err.name === "NotAllowedError" ? "denied" : "prompt",
        });
      }
    },
    [reduxDispatch, localStreamRef],
  );

  // stream watcher
  useEffect(() => {
    if (localStreamRef?.current && callType !== null) getDeviceStream(callType);
  }, [localStreamRef, callType, getDeviceStream]);

  const requestPermissions = useCallback(
    async (type: "voice" | "video") => {
      try {
        const stream = await navigator.mediaDevices.getUserMedia({
          video: type === "video",
          audio: true,
        });

        // Release tracks after permission granted
        stream.getTracks().forEach((track) => track.stop());

        dispatch({
          microphonePermission: "granted",
          cameraPermission:
            type === "video" ? "granted" : state.cameraPermission,
        });

        if (onCall) await getDeviceStream(type);
      } catch (error) {
        const err = error as DOMException;
        toast.error(`Permission denied: ${err.message}`);
        dispatch({
          microphonePermission: "denied",
          cameraPermission:
            type === "video" ? "denied" : state.cameraPermission,
        });
      }
    },
    [onCall, state.cameraPermission, getDeviceStream],
  );

  // Call management
  const endCall = useCallback(() => {
    console.log("Cleaning up media tracks");

    const stopAllTracks = (stream?: MediaStream | null) => {
      if (stream instanceof MediaStream) {
        stream.getTracks().forEach((track) => {
          try {
            track.stop();
          } catch (err) {
            console.warn("Error stopping track:", err);
          }
        });
      }
    };

    // Stop tracks BEFORE setting refs to null
    stopAllTracks(localStream.current);
    stopAllTracks(window.stream);
    stopAllTracks(localStreamRef?.current?.srcObject as MediaStream);

    // Stop explicitly referenced media tracks
    if (myAudioTrack.current) {
      myAudioTrack.current.stop();
      myAudioTrack.current = null;
      console.log("audio stopped");
    }

    if (myVideoTrack.current) {
      myVideoTrack.current.stop();
      myVideoTrack.current = null;
      console.log("video stopped");
    }

    if (myScreenTrack.current) {
      myScreenTrack.current.forEach((track) => track?.stop());
      myScreenTrack.current = [];
    }

    // Clear all refs
    localStream.current = null;

    if (localStreamRef?.current) {
      stopAllTracks(localStreamRef.current.srcObject as MediaStream);
      localStreamRef.current.srcObject = null;
    }

    // Redux + local cleanup
    reduxDispatch(setOnCall(false));
    reduxDispatch(setIncomingCall(null));
    reduxDispatch(setOutgoingCall(null));
    reduxDispatch(setLocalStream(false));
    reduxDispatch(setCurrentMeetingId(null));
    setUsersOnCall([]);

    window.location.reload();
  }, [reduxDispatch, localStreamRef]);

  // console.log("stream", localStream);
  // console.log("track", myVideoTrack);

  const onMakeCall = useCallback(
    async (type: "voice" | "video") => {
      if (onCall) {
        toast("You're already in a call");
        return;
      }

      reduxDispatch(setCallType(type));

      const needsCamera = type === "video";
      const requirements = [
        needsCamera && state.cameraPermission !== "granted" && "camera",
        state.microphonePermission !== "granted" && "microphone",
      ].filter(Boolean);

      if (requirements.length) {
        toast(`Please enable: ${requirements.join(" and ")}`);
        await requestPermissions(type);
      }

      try {
        reduxDispatch(setCallType(type));
        reduxDispatch(setOnCall(true));
        reduxDispatch(setCaller(true));
        reduxDispatch(
          setCurrentMeetingId(
            `${type === "video" ? "vd" : "vc"}-${generateMeetingId()}`,
          ),
        );

        await getDeviceStream(type);
      } catch (error) {
        console.error("Call setup failed:", error);
        toast.error("Failed to start call");
        endCall(); // Reset state on error
      }
    },
    [
      onCall,
      state.cameraPermission,
      state.microphonePermission,
      requestPermissions,
      reduxDispatch,
      getDeviceStream,
      endCall,
    ],
  );

  // Screen sharing

  const stopSharingScreen = useCallback(() => {
    myScreenTrack.current.forEach((track) => track?.stop());
    myScreenTrack.current = [];
    dispatch({ screenMode: false });

    // Restore original stream
    if (localStreamRef?.current && localStream.current) {
      localStreamRef.current.srcObject = localStream.current;
      localStreamRef.current.play().catch(console.error);
    }

    // Restore video tracks
    usersOnCall.forEach((user) => {
      const sender = user.peerCall.peerConnection
        .getSenders()
        .find((s) => s.track?.kind === "video");
      if (sender && myVideoTrack.current) {
        sender.replaceTrack(myVideoTrack.current);
      }
    });
  }, [localStream, usersOnCall, localStreamRef]);

  const shareScreen = useCallback(async () => {
    if (state.screenMode) return stopSharingScreen();

    try {
      const screenStream = await navigator.mediaDevices.getDisplayMedia({
        video: {
          cursor: "always",
          displaySurface: "window",
          width: { ideal: 1920 },
          height: { ideal: 1080 },
        },
        audio: true,
      });

      const [screenTrack] = screenStream.getVideoTracks();
      myScreenTrack.current = [screenTrack];
      dispatch({ screenMode: true });

      if (localStreamRef?.current) {
        localStreamRef.current.srcObject = screenStream;
        localStreamRef.current.play().catch(console.error);
      }

      screenTrack.onended = () => stopSharingScreen();

      // Update peer connections
      usersOnCall.forEach((user) => {
        const sender = user.peerCall.peerConnection
          .getSenders()
          .find((s) => s.track?.kind === "video");
        sender?.replaceTrack(screenTrack);
      });
    } catch (error) {
      toast.error(`Screen sharing failed: ${(error as Error).message}`);
      dispatch({ screenMode: false });
    }
  }, [state.screenMode, usersOnCall, stopSharingScreen, localStreamRef]);

  // Media controls
  const toggleMic = useCallback(() => {
    const audioTrack = myAudioTrack.current;
    if (audioTrack) {
      const newState = !audioTrack.enabled;
      audioTrack.enabled = newState;
      dispatch({ audioMode: newState });
    }
  }, [isMediaInitialized]); // Add dependency

  const toggleVideo = useCallback(() => {
    const videoTrack = myVideoTrack.current;
    if (videoTrack) {
      const newState = !videoTrack.enabled;
      videoTrack.enabled = newState;
      dispatch({ videoMode: newState });
    }
  }, [isMediaInitialized]);

  return {
    localStreamRef,
    remoteStreamRef,
    state,
    usersOnCall,
    onMakeCall,
    endCall,
    toggleMic,
    toggleVideo,
    shareScreen,
    stopSharingScreen,
    getDeviceStream,
    requestPermissions,
    localStream: localStream.current,
    myAudioTrack,
    myScreenTrack,
    myVideoTrack,
    // mediaTracks
  };
};

export default useCall;
