import * as React from "react";
import * as TooltipPrimitive from "@radix-ui/react-tooltip";
import { cn } from "@/utils/utilities";

const TooltipProvider = TooltipPrimitive.Provider;

const Tooltip = TooltipPrimitive.Root;

const TooltipTrigger = React.forwardRef<
  React.ElementRef<typeof TooltipPrimitive.Trigger>,
  React.ComponentPropsWithoutRef<typeof TooltipPrimitive.Trigger> & {
    asChild?: boolean;
    disableHoverableContent?: boolean;
  }
>(
  (
    { className, asChild = false, disableHoverableContent = false, ...props },
    ref,
  ) => {
    const [isOpen, setIsOpen] = React.useState(false);

    const handleClick = (e: React.MouseEvent) => {
      if (disableHoverableContent) {
        e.preventDefault();
        setIsOpen(!isOpen);
      }
      props.onClick?.(e as any);
    };

    return (
      <TooltipPrimitive.Trigger
        ref={ref}
        className={cn("cursor-pointer", className)}
        onClick={handleClick}
        data-state={isOpen ? "open" : "closed"}
        {...props}
      />
    );
  },
);
TooltipTrigger.displayName = TooltipPrimitive.Trigger.displayName;

const TooltipContent = React.forwardRef<
  React.ElementRef<typeof TooltipPrimitive.Content>,
  React.ComponentPropsWithoutRef<typeof TooltipPrimitive.Content> & {
    hideOnClick?: boolean;
  }
>(({ className, sideOffset = 4, hideOnClick = true, ...props }, ref) => {
  const [isVisible, setIsVisible] = React.useState(true);

  React.useEffect(() => {
    if (!isVisible && hideOnClick) {
      const timer = setTimeout(() => setIsVisible(true), 100);
      return () => clearTimeout(timer);
    }
  }, [isVisible, hideOnClick]);

  const handleClick = () => {
    if (hideOnClick) {
      setIsVisible(false);
    }
  };

  if (!isVisible) return null;

  return (
    <TooltipPrimitive.Content
      ref={ref}
      sideOffset={sideOffset}
      onClick={handleClick}
      className={cn(
        "animate-in fade-in-0 zoom-in-95 data-[state=closed]:animate-out data-[state=closed]:fade-out-0 data-[state=closed]:zoom-out-95 data-[side=bottom]:slide-in-from-top-2 data-[side=left]:slide-in-from-right-2 data-[side=right]:slide-in-from-left-2 data-[side=top]:slide-in-from-bottom-2 z-50 overflow-hidden rounded-md bg-white px-3 py-1.5 text-sm shadow-md",
        className,
      )}
      {...props}
    />
  );
});
TooltipContent.displayName = TooltipPrimitive.Content.displayName;

export { Tooltip, TooltipTrigger, TooltipContent, TooltipProvider };
