"use client";

import React, { useEffect, useState } from "react";
import AccountSelectTab from "./AccountSelectTab";
import { useDispatch, useSelector } from "react-redux";
import { AppDispatch, RootState } from "@/redux/store";
import { User, UserInstitution, UserRole } from "@/interfaces";
import {
  getToken,
  getUserData,
  setSessionActiveInstitutionId,
  setSessionActiveRole,
  setSessionUserData,
  setToken,
} from "@/utils/sessions/session.utils";
import {
  setActiveInstitution,
  setActiveRole,
  setActiveUser,
  setInstitutionAdmin,
  setPermissions,
} from "@/redux/user/userSlice";
import institutionService from "@/services/institution.service";
import toast from "react-hot-toast";
import { useRouter } from "next/navigation";
import Loading from "@/components/Loading/Loading";
// import pathologistService from "@/services/pathologist.service";

const AccountSelect = () => {
  const router = useRouter();
  const reduxDispatch = useDispatch<AppDispatch>();
  const { activeUser } = useSelector((state: RootState) => state.user);

  const [isLoading, setIsLoading] = useState<boolean>(false);

  useEffect(() => {
    const storedUser = getUserData();
    const storedToken = getToken();

    if (!storedToken) router.push("/auth/login");
    if (storedUser) {
      reduxDispatch(setActiveUser(storedUser));
    }
  }, []);

  const getInstitutionPathologistUserDetails = async (
    institutionId: number,
    role: UserRole,
  ) => {
    institutionService
      .getInstitutionPathologistUserDetails({ institutionId: institutionId })
      .then((res) => {
        // console.log(res.data);
        if (res?.data && activeUser) {
          const updatedUser = {
            ...activeUser,
            pathologist: {
              ...res.data,
              ...activeUser.pathologist,
              id: activeUser.pathologist?.id,
            },
          };

          setSessionUserData(updatedUser as User);
          reduxDispatch(setActiveUser(updatedUser as User));
        }
        onHandlePageNavigation(role);
      })
      .catch((error) => {
        console.log(error);
        toast.error(error?.message || error?.errors);
      })
      .finally(() => setIsLoading(false));
  };

  const onLogIntoInstitution = async (
    institutionId: number,
    role: UserRole,
  ) => {
    setIsLoading(true);
    institutionService
      .logIntoInstitution({
        institutionId: institutionId,
        loginAs: role,
      })
      .then((res) => {
        console.log("res", res);
        const {
          id,
          institution,
          user,
          token,
          institutionAdmin,
          institutionPathologist,
          ...permissions
        } = res.data as any;
        if (res.data) {
          setToken(token as string);

          reduxDispatch(setActiveInstitution(institution));
          reduxDispatch(setInstitutionAdmin(institutionAdmin));
          reduxDispatch(setPermissions(permissions));
          reduxDispatch(setActiveRole(role));

          setSessionActiveInstitutionId(institution.id);
          setSessionActiveRole(role);

          if (role === "PATHOLOGIST") {
            getInstitutionPathologistUserDetails(institutionId, role);
          } else {
            onHandlePageNavigation(role);
          }
        }

        toast.success(res.message);
      })
      .catch((error) => {
        console.log(error);
        toast.error(error?.message || error?.errors);
        setIsLoading(false);
      });
  };

  const onLogIntoPersonal = () => {
    setIsLoading(true);

    setTimeout(() => {
      toast.success("logged into personal");

      router.push("/personal/dashboard");
      reduxDispatch(setActiveRole("PATHOLOGIST"));
      setSessionActiveRole("PATHOLOGIST");

      setIsLoading(false);
    }, 1500);

    // pathologistService
    //   .getPathologistDetails({
    //     pathologistId: pathologistId,
    //   })
    //   .then((res) => {
    //     console.log("res", res);

    //     toast.success("logged into personal");
    //      router.push("/personal/dashboard");
    //   })
    //   .catch((error) => {
    //     console.log(error);
    //     toast.error(error?.message || error?.errors);
    //   })
    //   .finally(() => {
    //     setIsLoading(false);
    //   });
  };

  const onHandlePageNavigation = (role: UserRole) => {
    if (["SUPERADMIN", "ADMIN"].includes(role)) {
      router.push("/institution/dashboard");
    } else {
      router.push("/institution/view-studies");
    }
  };

  return (
    <div className="mx-auto flex max-h-[80vh] w-[85vw] max-w-[800px] flex-col rounded-[8px] bg-white p-10 shadow-md sm:w-[60vw]">
      <h2 className="mb-0 text-center text-[18px] font-semibold text-blackMain">
        Select Account
      </h2>
      <p className="mb-0 text-center text-[14px] text-grey">
        Please choose the account profile you want to login with.
      </p>
      <div className="relative mt-3 flex flex-1 overflow-hidden">
        {isLoading && <Loading className={{ loader: "!w-[50px]" }} />}
        <div className="mt-5 flex flex-1 flex-wrap items-center justify-center gap-8 overflow-y-auto">
          {activeUser?.pathologist?.id && (
            <AccountSelectTab
              role="PATHOLOGIST"
              accountName="Personal"
              onClick={() => onLogIntoPersonal()}
            />
          )}
          {activeUser?.userInstitutions?.map(
            (userInst: UserInstitution, idx) => (
              <React.Fragment key={idx}>
                <AccountSelectTab
                  role={userInst.role}
                  accountName={userInst.institution.institutionName}
                  logo={userInst.institution.logo}
                  onClick={() =>
                    onLogIntoInstitution(userInst.institution.id, userInst.role)
                  }
                />
              </React.Fragment>
            ),
          )}
        </div>
      </div>
    </div>
  );
};

export default AccountSelect;
