"use client";

import { CellContext, ColumnDef, Row, Table } from "@tanstack/react-table";
import { Checkbox } from "../ui/checkbox";
import React from "react";

// Generic interface for column configuration
export interface ColumnConfig<TData> {
  key: string;
  header?:
    | string
    | React.ReactNode
    | (({ table }: { table: Table<TData> }) => React.ReactNode);
  cell?: (context: CellContext<TData, unknown>) => string | React.ReactNode;
  type?: "text" | "currency" | "date" | "custom";
  currency?: string;
  dateFormat?: "short" | "long" | "full";
}

// Options for column generation
export interface ColumnGenerationOptions {
  select?: boolean;
  serialNumber?: boolean;
  customColumns?: any[];
}

export function generateColumns<TData>(
  dataColumns: ColumnConfig<TData>[],
  options: ColumnGenerationOptions = {},
): ColumnDef<TData>[] {
  const { select = false, serialNumber = false, customColumns = [] } = options;

  // Base columns array to be returned
  const columns: ColumnDef<TData>[] = [];

  // Optional Select Column
  if (select) {
    columns.push({
      id: "select",
      header: ({ table }: { table: Table<TData> }) => (
        <div className="px-2">
          <Checkbox
            className="border-grey"
            checked={
              table.getIsAllPageRowsSelected() ||
              (table.getIsSomePageRowsSelected() && "indeterminate")
            }
            onCheckedChange={(value) =>
              table.toggleAllPageRowsSelected(!!value)
            }
            aria-label="Select all"
          />
        </div>
      ),
      cell: ({ row }: { row: Row<TData> }) => (
        <div className="px-2">
          <Checkbox
            className="border-grey"
            checked={row.getIsSelected()}
            onCheckedChange={(value) => row.toggleSelected(!!value)}
            aria-label="Select row"
          />
        </div>
      ),
      enableSorting: false,
      enableHiding: false,
    });
  }

  // Optional Serial Number Column
  if (serialNumber) {
    columns.push({
      accessorKey: "sn",
      header: () => <div className="px-2 text-left">S/N</div>,
      cell: ({ row }: { row: Row<TData> }) => {
        const snValue = row.index + 1;
        return <div className="px-2 text-left font-medium">{snValue}</div>;
      },
    });
  }

  // Dynamic Column Generation
  const dynamicColumns = dataColumns.map((columnConfig): ColumnDef<TData> => {
    return {
      accessorKey: columnConfig.key,
      header:
        typeof columnConfig.header == "function"
          ? columnConfig.header
          : () => {
              // Handle header as string or function
              if (typeof columnConfig.header === "function") {
                return columnConfig.header(
                  {} as {
                    table: Table<TData>;
                  },
                );
              }

              if (
                typeof columnConfig.header === "string" ||
                React.isValidElement(columnConfig.header)
              ) {
                return (
                  <div className="whitespace-nowrap text-left">
                    {columnConfig.header}
                  </div>
                );
              }

              return columnConfig.header ?? null; // Handles JSX elements or null
            },
      cell:
        typeof columnConfig.cell == "function"
          ? columnConfig.cell
          : ({ row }) => {
              const value = row.getValue(columnConfig.key);
              if (columnConfig.cell) {
                return (
                  <div className="flex text-left font-medium">
                    {columnConfig.cell(
                      row.original as CellContext<TData, unknown>,
                    )}
                  </div>
                );
              }

              // Type-based formatting
              switch (columnConfig.type) {
                case "currency":
                  const formatted = new Intl.NumberFormat("en-US", {
                    style: "currency",
                    currency: columnConfig.currency || "USD",
                  }).format(value as number);
                  return (
                    <div className="text-left font-medium">{formatted}</div>
                  );

                case "date":
                  const date = new Date(value as string);
                  let formattedDate;
                  switch (columnConfig.dateFormat) {
                    case "short":
                      formattedDate = date.toLocaleDateString();
                      break;
                    case "full":
                      formattedDate = date.toLocaleDateString("en-US", {
                        year: "numeric",
                        month: "long",
                        day: "numeric",
                      });
                      break;
                    default:
                      formattedDate = date.toLocaleDateString();
                  }
                  return (
                    <div className="text-left font-medium">{formattedDate}</div>
                  );

                default:
                  return (
                    <div className="flex w-full flex-nowrap text-nowrap text-left font-medium">
                      {value as string}
                    </div>
                  );
              }
            },
    };
  });

  // Add custom columns if provided
  columns.push(...dynamicColumns, ...customColumns);

  return columns;
}
