import React from "react";
import { Table } from "@tanstack/react-table";
import {
  LuChevronLeft,
  LuChevronRight,
  LuChevronsLeft,
  LuChevronsRight,
} from "react-icons/lu";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "../ui/select";
import { Button } from "../ui/button";
import { cn } from "@/utils/utilities";

export default function DataTablePagination<TData>({
  table,
  totalPages = 1,
  handleChangeBatch = () => {},
  handleChangeLimit = () => {},
}: {
  table: Table<TData>;
  totalPages?: number;
  handleChangeBatch?: (value: number) => void;
  handleChangeLimit?: (value: number) => void;
}) {
  // Provide default values to prevent potential undefined errors
  const pageCount = Math.max(1, totalPages);
  const currentPage = Math.min(
    table.getState().pagination.pageIndex,
    pageCount - 1,
  );

  const renderPageButtons = () => {
    const pageButtons = [];
    const maxVisiblePages = 5;

    // Calculate start and end pages with safety checks
    const startPage = Math.max(
      0,
      Math.min(
        currentPage - Math.floor(maxVisiblePages / 2),
        pageCount - maxVisiblePages,
      ),
    );
    const endPage = Math.min(pageCount - 1, startPage + maxVisiblePages - 1);

    for (let i = startPage; i <= endPage; i++) {
      pageButtons.push(
        <Button
          key={`page-${i}`}
          variant={currentPage === i ? "default" : "outline"}
          className={cn("h-8 w-8 p-0", {
            "bg-primaryMain text-white": currentPage === i,
          })}
          onClick={() => {
            // Ensure page index is within bounds
            const safePageIndex = Math.max(0, Math.min(i, pageCount - 1));
            table.setPageIndex(safePageIndex);
            handleChangeBatch(safePageIndex + 1); // Assume 1-based page numbering for API
          }}
        >
          {i + 1}
        </Button>,
      );
    }

    // Add ellipsis if there are more pages
    if (endPage < pageCount - 1) {
      pageButtons.push(
        <span key="ellipsis" className="text-muted-foreground">
          ...
        </span>,
      );
    }

    return pageButtons;
  };

  // Default page sizes with a fallback
  const PAGE_SIZES = [10, 20, 30, 40, 50];
  const currentPageSize = table.getState().pagination.pageSize;
  const validPageSize = PAGE_SIZES.includes(currentPageSize)
    ? currentPageSize
    : PAGE_SIZES[1]; // Default to first size if invalid

  return (
    <div className="flex items-center justify-between px-2">
      <div className="text-muted-foreground flex-1 truncate text-sm">
        {table.getFilteredSelectedRowModel().rows.length} of{" "}
        {table.getFilteredRowModel().rows.length} row(s) selected.
      </div>
      <div className="flex items-center space-x-6 lg:space-x-8">
        <div className="flex items-center space-x-2">
          <p className="truncate text-sm font-medium">Rows per page</p>
          <Select
            value={`${validPageSize}`}
            onValueChange={(value) => {
              const newPageSize = Number(value);
              table.setPageSize(newPageSize);
              handleChangeLimit(newPageSize);
            }}
          >
            <SelectTrigger className="h-8 w-[70px]">
              <SelectValue placeholder={validPageSize} />
            </SelectTrigger>
            <SelectContent side="top">
              {PAGE_SIZES.map((pageSize) => (
                <SelectItem key={pageSize} value={`${pageSize}`}>
                  {pageSize}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
        {/* <div className="flex w-[100px] items-center justify-center text-sm font-medium">
          Page {currentPage + 1} of {pageCount}
        </div> */}
        <div className="flex items-center space-x-2">
          <Button
            variant="outline"
            className="hidden h-8 w-8 p-0 lg:flex"
            onClick={() => {
              table.setPageIndex(0);
              handleChangeBatch(1);
            }}
            disabled={currentPage === 0 || table.getPageCount() === 0}
          >
            <span className="sr-only">Go to first page</span>
            <LuChevronsLeft />
          </Button>
          <Button
            variant="outline"
            className="h-8 w-8 p-0"
            onClick={() => {
              table.previousPage();
              handleChangeBatch(currentPage);
            }}
            disabled={currentPage === 0}
          >
            <span className="sr-only">Go to previous page</span>
            <LuChevronLeft />
          </Button>
          <div className="flex items-center space-x-1">
            {renderPageButtons()}
          </div>
          <Button
            variant="outline"
            className="h-8 w-8 p-0"
            onClick={() => {
              table.nextPage();
              handleChangeBatch(currentPage + 2);
            }}
            disabled={currentPage >= pageCount - 1}
          >
            <span className="sr-only">Go to next page</span>
            <LuChevronRight />
          </Button>
          <Button
            variant="outline"
            className="hidden h-8 w-8 p-0 lg:flex"
            onClick={() => {
              table.setPageIndex(pageCount - 1);
              handleChangeBatch(pageCount);
            }}
            disabled={
              currentPage >= pageCount - 1 || table.getPageCount() === 0
            }
          >
            <span className="sr-only">Go to last page</span>
            <LuChevronsRight />
          </Button>
        </div>
      </div>
    </div>
  );
}
