"use client";

import {
  ColumnDef,
  ColumnFiltersState,
  flexRender,
  getCoreRowModel,
  getPaginationRowModel,
  SortingState,
  useReactTable,
} from "@tanstack/react-table";

import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import DataTablePagination from "./DataTablePagination";
import { useMemo, useState, useTransition } from "react";
import { cn } from "@/utils/utilities";
import { Input } from "../ui/input";
import { LuLoaderCircle } from "react-icons/lu";

interface DataTableProps<TData, TValue> {
  columns: ColumnDef<TData, TValue>[];
  data: TData[];
  theme?: "primary" | "dark";
  search?: {
    fields?: string[];
    placeholder: string;
    searchValue: string;
    handleChange: (e: React.ChangeEvent<HTMLInputElement>) => void;
  };
  stickLeft?: number[];
  stickRight?: number[];
  Loader?: { isLoading: boolean; loaderText: string };
  onRowClick?: (data: TData) => void;
}

const baseStyle = {
  header: "border-0",
  body: "border-0 gap-4 space-y-4 w-full h-full",
};

const variantStyle = (theme: string) => {
  switch (theme) {
    case "primary":
      return {
        header: "bg-greyFill text-blackMain [&_tr]:border-b-0 py-2",
        body: "text-grey bg-white",
      };
    case "dark":
      return {
        header: "b text-[#EDEEEE] [&_tr]:border-b-0",
        body: "border-0 bg-[#1B1E2B] text-[#9AA0A1]",
      };
    default:
      return {
        header: "",
        body: "",
      };
  }
};

export function DataTable<TData, TValue>({
  columns,
  data,
  theme = "primary",
  search,
  stickLeft,
  stickRight,
  Loader,
  onRowClick,
}: DataTableProps<TData, TValue>) {
  const memoizedColumns = useMemo(() => {
    if (!Array.isArray(columns)) return [];
    return columns;
  }, [columns]);

  const memoizedData = useMemo(() => {
    if (!Array.isArray(data)) return [];
    return data;
  }, [data]);
  const [sorting, setSorting] = useState<SortingState>([]);
  const [columnFilters, setColumnFilters] = useState<ColumnFiltersState>([]);
  const [isPending, startTransition] = useTransition();

  const table = useReactTable({
    data: memoizedData,
    columns: memoizedColumns,
    getCoreRowModel: getCoreRowModel(),
    getPaginationRowModel: getPaginationRowModel(),
    onColumnFiltersChange: setColumnFilters,
    manualFiltering: true,
    state: {
      sorting,
      columnFilters,
    },
  });

  return (
    <div className="w-full rounded-md bg-whiteShade [&_tr:hover]:bg-gray-50">
      <div className="flex justify-between px-4 py-4">
        <div className="flex items-center py-4"></div>
        {search && (
          <Input
            onChange={(e) => search.handleChange(e)}
            placeholder={search.placeholder}
            value={search.searchValue}
            className="w-[350px]"
          ></Input>
        )}
      </div>
      <Table className="mb-4 w-full">
        <TableHeader className={cn(variantStyle(theme)?.header)}>
          {table.getHeaderGroups().map((headerGroup) => (
            <TableRow className="border-0" key={headerGroup.id}>
              {headerGroup.headers.map((header, index) => {
                return (
                  <TableHead
                    className={cn(
                      "[&_tr]:border-b-0",
                      stickLeft?.includes(index) && "sticky left-0 z-10",
                      stickRight?.includes(index) && "sticky right-0 z-10",
                      "px-4",
                      variantStyle(theme).header,
                    )}
                    key={header.id}
                  >
                    {header.isPlaceholder
                      ? null
                      : flexRender(
                          header.column.columnDef.header,
                          header.getContext(),
                        )}
                  </TableHead>
                );
              })}
            </TableRow>
          ))}
        </TableHeader>
        <TableBody
          className={cn("relative", baseStyle.body, variantStyle(theme)?.body)}
        >
          {Loader?.isLoading || isPending ? (
            <TableRow className="border-0">
              <TableCell
                colSpan={columns.length}
                className="h-full w-full py-20 text-center"
              >
                <div className="flex items-center justify-center gap-4 text-sm text-greySecondary">
                  {Loader?.loaderText && <span>{Loader.loaderText}</span>}
                  <LuLoaderCircle className="animate-spin" size={24} />
                </div>
              </TableCell>
            </TableRow>
          ) : memoizedData.length === 0 ? (
            <TableRow>
              <TableCell colSpan={columns.length} className="h-24 text-center">
                No data available
              </TableCell>
            </TableRow>
          ) : (
            table.getRowModel().rows.map((row) => (
              <TableRow
                className={cn("border-0", onRowClick && "cursor-pointer")}
                key={row.id}
                data-state={row.getIsSelected() && "selected"}
                onClick={() => {
                  if (!onRowClick) return;
                  onRowClick(row.original);
                }}
              >
                {row.getVisibleCells().map((cell, index) => (
                  <TableCell
                    className={cn(
                      "py-3",
                      stickRight?.includes(index) && "sticky right-0 z-10",
                      "px-4",
                      variantStyle(theme).body,
                      stickLeft?.includes(index) && "sticky left-0 z-10",
                      [0, 1].includes(index) && "!w-[80px]",
                    )}
                    key={cell.id}
                  >
                    {flexRender(cell.column.columnDef.cell, cell.getContext())}
                  </TableCell>
                ))}
              </TableRow>
            ))
          )}
        </TableBody>
      </Table>

      <DataTablePagination totalPages={table.getPageCount()} table={table} />
    </div>
  );
}
