import Link from "next/link";
import React, { useEffect, useRef, useState, ReactNode } from "react";
import { MdExpandMore } from "react-icons/md";
import {
  TooltipContent,
  TooltipProvider,
  TooltipRoot,
  TooltipTrigger,
} from "../Tooltip/Tooltip";
import { useSelector } from "react-redux";
import { RootState } from "@/redux/store";

interface Props {
  icon?: ReactNode;
  label: string;
  to?: string;
  children?: ReactNode;
  isActive?: boolean;
  isChildActive?: boolean;
  onClick?: () => void;
}

const MenuItems = ({
  children,
  label,
  icon,
  to,
  isActive,
  isChildActive,
  onClick,
}: Props) => {
  const [isExpanded, setIsExpanded] = useState<boolean>(false);
  const contentRef = useRef<HTMLUListElement>(null);

  const { sideBarClosed } = useSelector((state: RootState) => state.global);

  const handleExpand = () => {
    setIsExpanded(!isExpanded);
  };

  useEffect(() => {
    if (contentRef.current) {
      contentRef.current.style.height = isExpanded
        ? `${contentRef.current.scrollHeight}px`
        : "0px";
    }
  }, [isExpanded]);

  const isParentActive = isChildActive && !isExpanded; // Parent should have background when collapsed

  return (
    <li
      className={`cursor-pointer rounded-bl-[8px] rounded-tl-[8px] transition-all duration-500 ease-in-out ${isActive || isParentActive ? "bg-menuBackground" : ""} ${children ? (!isExpanded ? "hover:bg-menuHover" : "") : "hover:bg-menuHover"} text-greyDisabled`}
    >
      <TooltipProvider>
        <TooltipRoot>
          <TooltipTrigger asChild>
            <Link
              href={!children ? (to ? to : "#") : "#"}
              onClick={children ? handleExpand : onClick}
              className="flex items-center gap-[15px] px-5 py-3"
            >
              <span>{icon}</span>
              <span
                className={`whitespace-nowrap text-[16px] font-medium transition-all duration-500 ease-in-out ${sideBarClosed ? "max-w-0 overflow-hidden opacity-0" : "max-w-auto opacity-100"}`}
              >
                {label}
              </span>

              {children && (
                <span
                  className={`ml-auto transition-all duration-500 ease-in-out ${isExpanded ? "rotate-180" : ""} ${sideBarClosed ? "max-w-0 overflow-hidden opacity-0" : "max-w-auto opacity-100"}`}
                >
                  <MdExpandMore size={24} />
                </span>
              )}
            </Link>
          </TooltipTrigger>
          <TooltipContent
            className={`${sideBarClosed ? "flex h-[45px] items-center bg-menuHover px-4 text-[16px] font-medium text-greyDisabled" : "hidden"}`}
            side="right"
          >
            <span>{label}</span>
          </TooltipContent>
        </TooltipRoot>
      </TooltipProvider>

      {children && (
        <ul
          ref={contentRef}
          className={`ml-1 flex flex-col gap-[2px] overflow-hidden transition-all duration-500 ease-in-out ${
            isExpanded ? "-mt-2 max-h-screen" : "max-h-0"
          }`}
        >
          {children}
        </ul>
      )}
    </li>
  );
};

export default MenuItems;
