"use client";

import React, { useMemo } from "react";
import toast from "react-hot-toast";
import {
  MdOutlineArrowBackIos,
  MdOutlineGroups,
  MdOutlinePhone,
  MdOutlineVideocam,
} from "react-icons/md";
import { Button } from "../Buttons";
import { usePathname, useRouter } from "next/navigation";
import { useDispatch, useSelector } from "react-redux";
import { AppDispatch, RootState } from "@/redux/store";
import {
  setDrawer,
  setDrawerChildren,
  setSideBarClosed,
} from "@/redux/global/globalSlice";
import { getInitials } from "@/utils/utilities";
import Image from "next/image";
import Logo from "@/assets/svg/xo-bridge-v3.svg";
import jamMenu from "@/assets/icons/jamMenu.svg";
import useCall from "@/hooks/useCall";

const Navbar = () => {
  const router = useRouter();
  const pathname = usePathname();
  const reduxDispatch = useDispatch<AppDispatch>();

  const showBackButton = [
    "/institution/view-studies/",
    "/institution/pathologists/",
    "/personal/view-studies/",
  ].some((path) => pathname.startsWith(path));

  const handleContacts = () => {
    reduxDispatch(setDrawerChildren({ key: "contacts" }));
    reduxDispatch(setDrawer({ isOpen: true, width: "500px" }));
  };

  const { sideBarClosed } = useSelector((state: RootState) => state.global);
  const { slideDetail } = useSelector((state: RootState) => state.data);
  const { activeInstitution, activeUser } = useSelector(
    (state: RootState) => state.user,
  );

  const { onMakeCall } = useCall();

  const navTitle = useMemo(() => {
    const studyViewerRegex =
      /^\/(institution|personal)\/view-studies\/[^/]+\/viewer\/[^/]+$/;

    const isStudyViewer = studyViewerRegex.test(pathname);

    return isStudyViewer
      ? slideDetail?.patientName
      : activeInstitution?.institutionName;
  }, [activeInstitution, slideDetail, pathname]);

  return (
    <div className="sticky top-0 z-20 flex w-full items-center justify-center bg-whiteShade shadow-md">
      {/* Large Devices */}
      <div className="flex h-[80px] w-full items-center justify-between px-5 max-lg:hidden">
        <div className="w-20">
          {showBackButton && (
            <Button
              variant="secondary"
              onClick={() => router.back()}
              className="border-none px-2 hover:bg-lightGreenLight"
            >
              <MdOutlineArrowBackIos size={24} className="text-primaryMain" />
            </Button>
          )}
        </div>
        <p className="max-w-[50%] truncate text-2xl font-semibold text-blackMain">
          {navTitle}
        </p>
        <div className="flex items-center gap-10 text-primary">
          <MdOutlinePhone
            size={24}
            className="cursor-pointer"
            onClick={() => onMakeCall("voice")}
          />
          <MdOutlineVideocam
            size={24}
            className="cursor-pointer"
            onClick={() => onMakeCall("video")}
          />
          <MdOutlineGroups
            size={24}
            className="cursor-pointer"
            onClick={handleContacts}
          />

          {activeInstitution?.logo ? (
            <Image
              className="mx-auto size-[45px] rounded-full border border-primary object-cover shadow-xl"
              src={activeInstitution?.logo}
              alt="institution logo"
              width={45}
              height={45}
              crossOrigin="anonymous"
            />
          ) : (
            <div className="flex size-[45px] items-center justify-center overflow-hidden rounded-full bg-primaryMain object-cover font-medium text-whiteShade">
              {getInitials(
                `${activeUser?.firstName} ${activeUser?.lastName}`,
              )}{" "}
            </div>
          )}
        </div>
      </div>

      {/* Mobile */}
      <div className="flex h-[80px] w-full items-center justify-between px-2 lg:hidden">
        {showBackButton ? (
          <div className="w-20">
            <Button
              variant="secondary"
              onClick={() => router.back()}
              className="border-none px-2 hover:bg-lightGreenLight"
            >
              <MdOutlineArrowBackIos size={24} className="text-primaryMain" />
            </Button>
          </div>
        ) : (
          <Image
            src={Logo}
            alt="logo"
            priority
            className={`h-[41px] w-[110px] object-contain`}
          />
        )}

        <p className="max-w-[50%] truncate text-xl font-semibold text-blackMain max-sm:hidden">
          {navTitle}
        </p>
        <div
          onClick={() => reduxDispatch(setSideBarClosed(!sideBarClosed))}
          className="button flex h-10 w-10 cursor-pointer items-center justify-center rounded-[6px] bg-lightGreenLight"
        >
          <Image
            src={jamMenu}
            alt="menu"
            priority
            aria-label="menu toggle switch to open menu"
          />
        </div>
      </div>
    </div>
  );
};

export default Navbar;
