"use client";

import React, { useState } from "react";
import { MdEmail, MdLocalPhone, MdLocationOn } from "react-icons/md";
import { FaLinkedinIn } from "react-icons/fa";
import { IoLogoFacebook } from "react-icons/io";
import CircleVector from "@/assets/icons/circle_vector.svg";
import Image from "next/image";
import { Field, Form, Formik } from "formik";
import Link from "next/link";
import userService from "@/services/user.service";
import toast from "react-hot-toast";
import { Button } from "@/components/Buttons";
import { VISITOR_SCHEMA } from "@/utils/validationSchema/common.schema";

interface FormValues {
  email: string;
  name: string;
  subject?: string;
  message?: string;
}

const initialValues: FormValues = {
  email: "",
  name: "",
  subject: "",
  message: "",
};

const ContactUs = () => {
  const [loading, setLoading] = useState<boolean>(false);

  const handleSubmit = (
    values: FormValues,
    { resetForm }: { resetForm: () => void },
  ) => {
    setLoading(true);
    userService
      .sendVisitorsMessage(values)
      .then((res) => {
        toast.success(res?.message);
        console.log(res);
        resetForm();
      })
      .catch((error) => {
        toast.error(error?.errors || error?.message);
        console.log(error);
      })
      .finally(() => setLoading(false));
  };
  return (
    <div id="contact" className="bg-contact-gradient">
      <div className="mx-auto flex max-w-[1280px] flex-col justify-center gap-[50px] px-4 py-20 md:flex-row">
        <div className="relative flex h-[470px] w-full max-w-full flex-col justify-between rounded-[18px] bg-primary p-10 md:h-[600px] md:w-[40%] md:max-w-[460px]">
          <div className="">
            <div>
              <h3 className="mb-2 text-3xl font-bold text-white md:text-[40px]">
                Contact us
              </h3>
              <p className="text-left text-sm font-medium text-whiteShade md:text-[16px]">
                Fill up the form and our friendly team will get back to you
                within 24hrs.
              </p>
            </div>
            <div className="mt-6 space-y-2 md:mt-10 md:space-y-5">
              <span className="flex items-center justify-start gap-[20px] text-sm font-normal text-white sm:text-[16px]">
                <MdLocalPhone className="text-lg md:text-2xl" />
                <p className="mb-0"> +234 904 000 4400</p>
              </span>
              <span className="flex items-center justify-start gap-[20px] text-sm font-normal text-white sm:text-[16px]">
                <MdEmail className="text-lg md:text-2xl" />
                <p className="mb-0">hello@xolanihealth.com</p>
              </span>
              <span className="flex items-center justify-start gap-[20px] text-sm font-normal text-white sm:text-[16px]">
                <MdLocationOn className="text-lg md:text-2xl" />
                <p className="mb-0">
                  {" "}
                  No. 5, Kwaji Close, Maitama - Abuja, Nigeria.
                </p>
              </span>
            </div>
          </div>
          <div className="z-10 flex gap-5 text-[#ebebeb]">
            <Link
              href="mailto:xolanihealth@gmail.com?subject=Hello"
              title="Send an Email"
            >
              <MdEmail size={20} />
            </Link>
            <Link
              href="https://web.facebook.com/xolanihealth"
              target="_blank"
              title="Follow us on Facebook"
            >
              <IoLogoFacebook size={20} />
            </Link>
            <Link
              href="https://www.linkedin.com/company/xolanihealth/posts/?feedView=all"
              target="_blank"
              title="Follow us on LinkedIn"
            >
              <FaLinkedinIn size={20} />
            </Link>
          </div>
          <Image
            src={CircleVector}
            alt="vector"
            className="absolute bottom-0 right-0 z-0 rounded-br-[18px] max-sm:w-[150px]"
            loading="lazy"
          />
        </div>

        <Formik
          initialValues={initialValues}
          validationSchema={VISITOR_SCHEMA}
          onSubmit={handleSubmit}
        >
          {({ errors, touched }) => (
            <Form className="mt-4 flex-1 space-y-2 py-6">
              <div>
                <label
                  className="text-[13px] font-medium text-grey"
                  htmlFor="name"
                >
                  Name
                </label>
                <Field
                  id="name"
                  name="name"
                  placeholder="Enter your name"
                  disabled={loading}
                  className="bg-bgInput mt-1 flex h-[50px] w-full items-center rounded-[8px] border border-greyLine pl-3 pr-6 text-[14px] text-blackMain outline-none"
                />
                {errors.name && touched.name ? (
                  <p className="mb-0 text-[12px] font-medium text-[#ff0000]">
                    {errors.name}
                  </p>
                ) : null}
              </div>
              <div>
                <label
                  className="text-[13px] font-medium text-grey"
                  htmlFor="email"
                >
                  Email
                </label>
                <Field
                  id="email"
                  name="email"
                  placeholder="Enter your email"
                  disabled={loading}
                  className="bg-bgInput mt-1 flex h-[50px] w-full items-center rounded-[8px] border border-greyLine pl-3 pr-6 text-[14px] text-blackMain outline-none"
                />
                {errors.email && touched.email ? (
                  <p className="mb-0 text-[12px] font-medium text-[#ff0000]">
                    {errors.email}
                  </p>
                ) : null}
              </div>
              <div>
                <label
                  className="text-[13px] font-medium text-grey"
                  htmlFor="subject"
                >
                  Subject
                </label>
                <Field
                  id="subject"
                  name="subject"
                  placeholder="Enter appropriate title for your message"
                  disabled={loading}
                  className="bg-bgInput mt-1 flex h-[50px] w-full items-center rounded-[8px] border border-greyLine pl-3 pr-6 text-[14px] text-blackMain outline-none"
                />
                {errors.subject && touched.subject ? (
                  <p className="mb-0 text-[12px] font-medium text-[#ff0000]">
                    {errors.subject}
                  </p>
                ) : null}
              </div>
              <div>
                <label
                  className="text-[13px] font-medium text-grey"
                  htmlFor="message"
                >
                  Message
                </label>
                <Field
                  id="message"
                  name="message"
                  as="textarea"
                  rows={6}
                  placeholder="Leave us a message"
                  disabled={loading}
                  className="bg-bgInput mt-1 flex w-full items-center rounded-[8px] border border-greyLine py-2 pl-3 pr-6 text-[14px] text-blackMain outline-none"
                />
                {errors.message && touched.message ? (
                  <p className="mb-0 text-[12px] font-medium text-[#ff0000]">
                    {errors.message}
                  </p>
                ) : null}
              </div>

              <div className="flex justify-end">
                <Button
                  variant="primary"
                  className="mt-8 h-[50px]"
                  loading={loading}
                >
                  Send Your Message
                </Button>
              </div>
            </Form>
          )}
        </Formik>
      </div>
    </div>
  );
};

export default ContactUs;
