"use client";

import React, { useState } from "react";
import { Field, FieldProps, getIn } from "formik";
import { cn } from "@/utils/utilities";
import { BsEye, BsEyeSlash } from "react-icons/bs";
import { MdOutlineSearch } from "react-icons/md";

interface TextProps {
  name: string;
  type: "text" | "email" | "password" | "textarea" | "search";
  label?: string;
  placeholder?: string;
  className?: string;
  disabled?: boolean;
  loading?: boolean;
  rows?: number;
  onChange?: (
    e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>,
  ) => void;
}

export const TextInput = ({
  name,
  type,
  label,
  placeholder,
  className,
  disabled = false,
  loading = false,
  rows = 5,
  onChange,
}: TextProps) => {
  const [showPassword, setShowPassword] = useState<boolean>(false);

  return (
    <Field name={name}>
      {({ field, form }: FieldProps) => {
        const error = getIn(form.errors, field.name);
        const touched = getIn(form.touched, field.name);

        const handleChange = (
          e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>,
        ) => {
          if (onChange) {
            onChange(e); // Call custom onChange handler if it exists
          }
          field.onChange(e); // Call Formik's internal onChange handler
        };

        const renderInput = () => {
          switch (type) {
            case "email":
              return (
                <div className="relative flex items-center">
                  <input
                    {...field}
                    type="email"
                    placeholder={placeholder}
                    disabled={disabled || loading}
                    className={cn(
                      className,
                      "bg-bgInput flex h-[40px] w-full items-center rounded-[4px] border border-greyLine pl-3 pr-6 text-[14px] text-blackMain outline-none",
                    )}
                    onChange={handleChange}
                  />
                </div>
              );
            case "password":
              return (
                <div className="relative flex items-center">
                  <input
                    {...field}
                    type={showPassword ? "text" : "password"}
                    placeholder={placeholder}
                    disabled={disabled || loading}
                    className={cn(
                      className,
                      "bg-bgInput flex h-[40px] w-full items-center rounded-[4px] border border-greyLine pl-3 pr-6 text-[14px] text-blackMain outline-none",
                    )}
                    onChange={handleChange}
                  />
                  {showPassword ? (
                    <BsEyeSlash
                      onClick={() => setShowPassword(!showPassword)}
                      size={18}
                      className="absolute right-2 cursor-pointer text-greyLight"
                    />
                  ) : (
                    <BsEye
                      onClick={() => setShowPassword(!showPassword)}
                      size={18}
                      className="absolute right-2 cursor-pointer text-greyLight"
                    />
                  )}
                </div>
              );
            case "textarea":
              return (
                <div className="relative flex items-center">
                  <textarea
                    {...field}
                    rows={rows}
                    placeholder={placeholder}
                    disabled={disabled || loading}
                    className={cn(
                      className,
                      "bg-bgInput flex min-h-[40px] w-full items-center rounded-[4px] border border-greyLine p-3 text-[14px] text-blackMain outline-none",
                    )}
                    onChange={handleChange}
                  />
                </div>
              );
            case "search":
              return (
                <div className="relative flex items-center">
                  <input
                    {...field}
                    type={type}
                    placeholder={placeholder}
                    disabled={disabled || loading}
                    className={cn(
                      className,
                      "bg-bgInput flex h-[40px] w-full items-center rounded-[4px] border border-greyLine pl-8 pr-6 text-[14px] text-blackMain outline-none",
                    )}
                    onChange={handleChange}
                  />
                  <MdOutlineSearch
                    size={18}
                    className="absolute left-2 cursor-pointer text-greyLight"
                  />
                </div>
              );
            default:
              return (
                <input
                  {...field}
                  type={type}
                  placeholder={placeholder}
                  disabled={disabled || loading}
                  className={cn(
                    className,
                    "bg-bgInput flex h-[40px] w-full items-center rounded-[4px] border border-greyLine pl-3 pr-6 text-[14px] text-blackMain outline-none",
                  )}
                  onChange={handleChange}
                />
              );
          }
        };

        return (
          <div>
            {label && (
              <label
                htmlFor={name}
                className="text-[12px] font-medium text-grey"
              >
                {label}
              </label>
            )}
            {renderInput()}
            {touched && error && (
              <p className="mt-1 text-[10px] font-medium text-[#ff0000]">
                {error}
              </p>
            )}
          </div>
        );
      }}
    </Field>
  );
};
