"use client";

import React, { useRef, useCallback } from "react";
import { Field, FieldProps, getIn } from "formik";
import { cn } from "@/utils/utilities";
import { MdOutlineCalendarToday, MdOutlineAccessTime } from "react-icons/md";

interface DateTimeInputProps {
  name: string;
  type: "date" | "time";
  label?: string;
  placeholder?: string;
  className?: string;
  disabled?: boolean;
  loading?: boolean;
  min?: string;
  max?: string;
  step?: number;
  onChange?: (e: React.ChangeEvent<HTMLInputElement>) => void;
}

export const DateTimeInput = ({
  name,
  type,
  label,
  placeholder,
  className,
  disabled = false,
  loading = false,
  min,
  max,
  step,
  onChange,
}: DateTimeInputProps) => {
  const inputRef = useRef<HTMLInputElement | null>(null);
  const openPicker = useCallback((e?: React.MouseEvent) => {
    if (e) e.preventDefault();
    if (!inputRef.current) return;
    inputRef.current.focus();
    const anyInput = inputRef.current as unknown as {
      showPicker?: () => void;
      click: () => void;
    };
    try {
      anyInput.showPicker?.();
    } catch (err) {
      // Fallback when showPicker requires a direct user gesture
      anyInput.click();
    }
  }, []);
  return (
    <Field name={name}>
      {({ field, form }: FieldProps) => {
        const error = getIn(form.errors, field.name);
        const touched = getIn(form.touched, field.name);

        const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
          if (onChange) onChange(e);
          field.onChange(e);
        };

        const isEmpty = !field.value;

        return (
          <div>
            {label && (
              <label
                htmlFor={name}
                className="text-[12px] font-medium text-grey"
              >
                {label}
              </label>
            )}
            <div
              className="relative flex cursor-pointer items-center"
              onClick={openPicker}
              onMouseDown={openPicker}
            >
              <input
                {...field}
                id={name}
                type={type}
                placeholder={placeholder}
                disabled={disabled || loading}
                min={min}
                max={max}
                step={step}
                className={cn(
                  className,
                  "bg-bgInput flex h-[40px] w-full cursor-pointer items-center rounded-[4px] border border-greyLine pl-3 pr-8 text-[14px] outline-none",
                  isEmpty
                    ? "dt-input dt-input--empty text-greySecondary"
                    : "dt-input text-blackMain",
                )}
                onChange={handleChange}
                ref={inputRef}
              />
              {type === "date" ? (
                <MdOutlineCalendarToday
                  size={18}
                  aria-hidden
                  className="absolute right-2 cursor-pointer text-greyLine"
                  onClick={openPicker}
                />
              ) : (
                <MdOutlineAccessTime
                  size={18}
                  aria-hidden
                  className="absolute right-2 cursor-pointer text-greyLine"
                  onClick={openPicker}
                />
              )}
            </div>
            {touched && error && (
              <p className="mt-1 text-[10px] font-medium text-[#ff0000]">
                {error}
              </p>
            )}
          </div>
        );
      }}
    </Field>
  );
};
