"use client";
import React, { useState, useMemo } from "react";
import {
  Select,
  SelectContent,
  SelectGroup,
  SelectItem,
  SelectLabel,
  SelectSeparator,
  SelectTrigger,
  SelectValue,
} from "../ui/select";
import { ClassValue } from "clsx";
import { cn } from "@/utils/utilities";

export interface Options {
  label: React.ReactNode;
  value: string;
  separator?: boolean;
}

export interface DropdownProps {
  label?: string;
  options: Options[];
  className?: Styles;
  type?: "primary" | "dark";
  onValueChange: (value: string) => void;
  disabled?: boolean;
  placeholder?: string;
  loading?: boolean;
  value?: string;
  align?: "start" | "end" | "center";
  search?: boolean;
}

interface Styles {
  trigger?: ClassValue;
  content?: ClassValue;
  item?: ClassValue;
  label?: ClassValue;
}

export default function Dropdown({
  label,
  options,
  className,
  type = "primary",
  disabled = false,
  loading = false,
  onValueChange,
  placeholder,
  value,
  align = "start",
  search = false,
}: DropdownProps) {
  const [searchQuery, setSearchQuery] = useState("");

  // Filtered options based on search query
  const filteredOptions = useMemo(
    () =>
      search
        ? options.filter((data) =>
            data.label
              ?.toString()
              .toLowerCase()
              .includes(searchQuery.toLowerCase()),
          )
        : options,
    [searchQuery, options, search],
  );

  const baseStyles: Styles = {
    content:
      "group rounded-[8px] font-medium text-[14px] transition-all duration-200 ease-in-out",
    trigger: "",
    item: "",
    label: "cursor-default select-none",
  };

  const variantStyle = (type: DropdownProps["type"]): Styles | undefined => {
    switch (type) {
      case "primary":
        return {
          content: "bg-whiteShade ",
          item: "hover:text-[#A4B1B2] focus:text-[#A4B1B2] cursor-pointer text-grey focus:text-blackMain",
          trigger: "bg-whiteShade text-grey",
          label: "text-[#A4B1B2] ",
        };
      case "dark":
        return {
          content: "border-none bg-backgroundAdmin3 focus:border-none",
          item: "hover:text-[#9AA0A1] focus:text-[#9AA0A1] cursor-pointer  text-greyLine focus:bg-backgroundAdmin3 focus:text-[#9AA0A1] ",
          trigger: "bg-backgroundAdmin3 text-[#9AA0A1]",
          label: "text-[#9AA0A1] ",
        };
    }
  };

  const selectedItem = useMemo(
    () => options.find((item) => item.value === value),
    [value, options],
  );

  return (
    <Select
      value={value}
      disabled={disabled || loading}
      onValueChange={onValueChange}
    >
      <SelectTrigger
        className={cn(
          baseStyles.trigger,
          variantStyle(type)?.trigger,
          className?.trigger,
          {
            "cursor-not-allowed opacity-50": disabled || loading,
          },
        )}
      >
        {search ? (
          <input
            type="text"
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            placeholder={placeholder || "Search..."}
            className="w-full bg-transparent outline-none"
            disabled={disabled || loading}
          />
        ) : value ? (
          selectedItem ? (
            selectedItem.label
          ) : (
            value
          )
        ) : (
          <SelectValue placeholder={placeholder} />
        )}
      </SelectTrigger>
      <SelectContent
        align={align}
        side="bottom"
        className={cn(
          baseStyles,
          variantStyle(type)?.content,
          className?.content,
        )}
      >
        <SelectGroup>
          {label && (
            <SelectLabel
              className={cn(
                baseStyles.label,
                variantStyle(type)?.label,
                className?.label,
              )}
            >
              {label}
            </SelectLabel>
          )}
          {filteredOptions.map((data) => (
            <React.Fragment key={`item-${data.value}`}>
              <SelectItem
                className={cn(
                  baseStyles.item,
                  variantStyle(type)?.item,
                  className?.item,
                )}
                value={data.value}
              >
                {data.label}
              </SelectItem>
              {data.separator && <SelectSeparator />}
            </React.Fragment>
          ))}
        </SelectGroup>
      </SelectContent>
    </Select>
  );
}
