import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuGroup,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { Fragment } from "react";
import { ClassValue } from "clsx";
import { cn } from "@/utils/utilities";

interface ActionDataProps {
  label?: string | React.ReactNode;
  icon?: React.ReactNode;
  onClick?: (e?: React.MouseEvent) => void;
  separator?: boolean;
  customComponent?: React.ReactNode; // Allow custom components
}

interface Styles {
  label?: ClassValue;
  trigger?: ClassValue;
  content?: ClassValue;
  item?: ClassValue;
}

export interface ActionDropdownProps {
  trigger: React.ReactNode;
  options: ActionDataProps[];
  label?: string;
  align?: "start" | "end" | "center";
  theme?: "primary" | "dark";
  className?: Styles;
}

export function ActionDropdown({
  trigger,
  options,
  label,
  align = "start",
  theme = "primary",
  className,
}: ActionDropdownProps) {
  const baseStyle: Styles = {
    content: "w-[227px] p-2 border-none",
    trigger: "cursor-pointer",
    item: "cursor-pointer text-xs",
    label: "",
  };

  const variantStyle = (theme: string): Styles | undefined => {
    switch (theme) {
      case "primary":
        return {
          content: "bg-[#FFF] shadow-xl",
          trigger: "cursor-pointer",
          item: "text-grey hover:bg-greyStroke",
          label: "",
        };
      case "dark":
        return {
          content: "bg-backgroundAdmin3",
          trigger: "cursor-pointer",
          item: "text-greyLine",
          label: "",
        };
      default:
        return;
    }
  };

  return (
    <DropdownMenu>
      <DropdownMenuTrigger
        className={cn(
          baseStyle.trigger,
          variantStyle(theme)?.trigger,
          className?.trigger,
        )}
        asChild
      >
        {trigger}
      </DropdownMenuTrigger>
      <DropdownMenuContent
        className={cn(
          baseStyle.content,
          variantStyle(theme)?.content,
          className?.content,
        )}
        align={align}
      >
        {label && (
          <>
            <DropdownMenuLabel
              className={cn(
                baseStyle.label,
                variantStyle(theme)?.label,
                className?.label,
              )}
            >
              {label}
            </DropdownMenuLabel>
            <DropdownMenuSeparator />
          </>
        )}
        <DropdownMenuGroup>
          {options.map((item, idx) => (
            <Fragment
              key={typeof item.label === "string" ? item.label : `key-${idx}`}
            >
              {item.customComponent ? (
                item.customComponent
              ) : (
                <DropdownMenuItem
                  className={cn(
                    baseStyle.item,
                    variantStyle(theme)?.item,
                    className?.item,
                  )}
                  onClick={(e) => item.onClick?.(e)}
                >
                  {item.icon && item.icon}
                  <span>{item.label}</span>
                </DropdownMenuItem>
              )}
              {item.separator && <DropdownMenuSeparator />}
            </Fragment>
          ))}
        </DropdownMenuGroup>
      </DropdownMenuContent>
    </DropdownMenu>
  );
}
