import { cn } from "@/utils/utilities";
import React from "react";

interface ButtonProps extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  variant:
    | "primary"
    | "secondary"
    | "gray"
    | "gradient"
    | "lightGreen"
    | "normal"
    | "icon";
  children: React.ReactNode;
  className?: string;
  type?: "button" | "submit" | "reset";
  as?: React.ElementType;
  onClick?: () => void;
  loading?: boolean;
  disabled?: boolean;
  iconPrefix?: React.ReactNode;
  iconSuffix?: React.ReactNode;
  title?: string;
}

export const Button = ({
  variant = "primary",
  type,
  children,
  className,
  as: Component = "button",
  onClick,
  loading = false,
  disabled = false,
  iconPrefix = null,
  iconSuffix = null,
  title,
  ...props
}: ButtonProps) => {
  const baseStyles =
    "group button px-8 rounded-[8px] font-semibold text-[14px] flex items-center justify-center gap-2   outline-none transition-all  duration-200 ease-in-out h-[40px]";

  const variantStyle = (variant: ButtonProps["variant"]) => {
    switch (variant) {
      case "primary":
        return "bg-primaryMain text-white hover:bg-primary";
      case "secondary":
        return "bg-white border border-primaryMain text-primaryMain hover:bg-[#e6e6e6]";
      case "gray":
        return "bg-greyFill text-grey border border-greyStroke hover:bg-[#d9d9d9]";
      case "gradient":
        return "bg-buttonGradient text-white";
      case "lightGreen":
        return "bg-lightGreenLight border border-lightGreenLight text-primaryMain hover:border-lightGreen ";
      case "normal":
        return "bg-white  text-primaryMain border-none px-2 hover:bg-lightGreenLight ";
      case "icon":
        return " border-none p-0 ";
      default:
        return "";
    }
  };

  const loaderClass = (variant: ButtonProps["variant"]) => {
    switch (variant) {
      case "primary":
        return "loader-primary";
      case "secondary":
        return "loader-secondary";
      case "gradient":
        return "loader-gradient";
      case "gray":
        return "loader-grey";
      case "lightGreen":
        return "loader-secondary";
      case "normal":
        return "loader-secondary";
      default:
        return "";
    }
  };

  return (
    <Component
      onClick={onClick}
      disabled={disabled || loading}
      type={type}
      className={cn(
        baseStyles,
        variantStyle(variant),
        disabled ? "cursor-not-allowed opacity-50" : "",
        className,
      )}
      title={title}
      {...props}
    >
      <React.Fragment>
        {iconPrefix && !iconSuffix && <span>{iconPrefix}</span>}
        <span>{children}</span>
        {loading && (
          <span
            className={cn("loader ml-1", loaderClass(variant))}
            aria-hidden="true"
          ></span>
        )}
        {iconSuffix && !iconPrefix && <span>{iconSuffix}</span>}
      </React.Fragment>
    </Component>
  );
};
