import { setSlides } from "@/redux/data/dataSlice";
import { AppDispatch, RootState } from "@/redux/store";
import slideService from "@/services/slide.service";
import { useCallback, useEffect, useRef, useState } from "react";
import toast from "react-hot-toast";
import { useDispatch, useSelector } from "react-redux";

interface SearchState {
  uploaded: {
    placeholder: string;
    searchValue: string;
    fields: string[];
  };
  assigned: {
    placeholder: string;
    searchValue: string;
    fields: string[];
  };
}

interface SearchParams {
  searchValue: string;
  fields: string[];
}
const useViewStudies = () => {
  const reduxDispatch = useDispatch<AppDispatch>();
  const timeoutRef = useRef<NodeJS.Timeout | null>(null);
  const [loadingUploaded, setLoadingUploaded] = useState<boolean>(false);

  const { activeUser } = useSelector((state: RootState) => state.user);
  const { slides } = useSelector((state: RootState) => state.data);

  const [search, setSearch] = useState<SearchState>({
    uploaded: {
      placeholder: "Search by Name...",
      searchValue: "",
      fields: ["patientName", "modality"],
    },
    assigned: {
      placeholder: "Search by Name...",
      searchValue: "",
      fields: ["patientName", "modality"],
    },
  });

  // Create ref for active User
  const activeUserRef = useRef(activeUser);

  // Keep ref updated
  useEffect(() => {
    activeUserRef.current = activeUser;
  }, [activeUser]);

  const getSlides = useCallback(
    (searchParams: SearchParams) => {
      const currentUser = activeUserRef.current;
      if (!currentUser?.pathologist?.id) {
        toast.error("No active pathologist selected");
        return;
      }

      if (!slides.length) setLoadingUploaded(true);
      if (!!searchParams.searchValue) setLoadingUploaded(true);

      slideService
        .getPathologistSlides({
          pathologistId: currentUser?.pathologist?.id,
          ...(searchParams && {
            search: searchParams.searchValue || "",
            searchFields: searchParams.fields,
          }),
        })
        .then((res) => {
          reduxDispatch(setSlides(res?.data as any[]));
        })
        .catch((error) => {
          console.log(error);
          toast.error(error?.message || error?.errors);
          setLoadingUploaded(false);
        })
        .finally(() => setLoadingUploaded(false));
    },
    [reduxDispatch, slides.length],
  );

  const handleSearch = (
    e: React.ChangeEvent<HTMLInputElement>,
    type: keyof SearchState,
  ) => {
    const value = e.target.value;

    setSearch((prev) => {
      const updatedSearch = {
        ...prev,
        [type]: {
          ...prev[type],
          searchValue: value,
        },
      };

      // Clear existing timeout
      if (timeoutRef.current) {
        clearTimeout(timeoutRef.current);
      }

      // Set new debounced search
      timeoutRef.current = setTimeout(() => {
        if (type === "uploaded") {
          getSlides(updatedSearch.uploaded);
        }
      }, 500); // Optimal debounce time

      return updatedSearch;
    });
  };

  return {
    getSlides,
    loadingUploaded,
    handleSearch,
    search,
  };
};

export default useViewStudies;
