import { User } from "@/interfaces";
import {
  Document,
  Page,
  Text,
  View,
  StyleSheet,
  Image,
  pdf,
} from "@react-pdf/renderer";
import moment from "moment";
import React from "react";
import { PDFDocument, rgb, StandardFonts } from "pdf-lib";
import { extractDimensions } from "@/utils/utilities";

const styles = StyleSheet.create({
  topContainer: {
    width: "100%",
    alignSelf: "center",
    flexDirection: "row",
    justifyContent: "space-between",
    alignItems: "center",
    borderBottomColor: "#999",
    borderBottomWidth: 1,
    paddingBottom: 5,
    paddingVertical: 5,
  },
  headerContainer: {
    width: "100%",
    alignSelf: "center",
    flexDirection: "row",
    justifyContent: "center",
    alignItems: "center",
    borderBottomColor: "#999",
    borderBottomWidth: 1,
    paddingBottom: 5,
    paddingVertical: 5,
  },

  institutionDetails: {
    fontSize: 12,
    textAlign: "center",
  },
  patientDetails: {
    fontSize: 12,
    width: "45%",
  },

  diagnosis: {
    marginBottom: 15,
    paddingBottom: 15,
    paddingVertical: 15,
  },
  heading: {
    fontWeight: "700",
    fontSize: 12,
  },
  // signature: {
  //   width: 150,
  //   height: 50,
  //   marginTop: 15,
  // },

  page: {
    padding: 30,
    fontSize: 12,
    color: "#4f5055",
    lineHeight: 1.6,
    position: "relative",
  },
  pageLetterHead: {
    // padding: 30,
    fontSize: 12,
    color: "#4f5055",
    lineHeight: 1.6,
    position: "relative",
  },

  noHeaderPage: {
    padding: 30,
    position: "absolute",
    width: "100%",
    top: 0,
    left: 0,
    marginBottom: 86,
    marginTop: 86,
    zIndex: 10000,
  },
  backgroundImage: {
    position: "relative",
    width: "100%",
    top: 0,
    left: 0,
    bottom: 0,
  },
  section: {
    marginBottom: 10,
  },
  noHeader: {
    marginBottom: 10,
    marginTop: 100,
  },
  row: {
    flexDirection: "row",
    justifyContent: "space-between",
    flexWrap: "wrap",
  },
  label: {
    fontWeight: "normal",
    fontSize: 10,
    marginBottom: -5,
    paddingBottom: 0,
  },
  text: {
    fontWeight: "semibold",
    fontSize: 11,
    lineHeight: 1.2,
  },
  textBlock: {
    marginBottom: 10,
    minHeight: 50,
  },
  mainSection: {
    marginTop: 15,
  },
  signature: {
    width: 150,
    height: 50,
    marginTop: 10,
  },
  underline: {
    borderBottomWidth: 1,
    borderBottomColor: "#999",
    marginBottom: 10,
    width: "100%",
  },
});

const ReportDocument = ({
  activeUser,
  slideDetail,
  report,
}: {
  activeUser: User;
  slideDetail: any;
  report: { diagnosis: string; microscopy: string; recommendation: string };
}) => {
  const getAge = (birthDate: Date) => {
    return birthDate
      ? moment().diff(moment(birthDate, "YYYYMMDD"), "years")
      : "NA";
  };

  return (
    <Document title="Report Preview">
      <Page size="A4" style={styles.page}>
        {/* Institution Header */}
        <View style={styles.section}>
          <Text
            style={{ textAlign: "center", fontSize: 14, fontWeight: "bold" }}
          >
            {`${activeUser?.firstName || ""} ${activeUser?.lastName || " Report"}`}
          </Text>
          <Text style={{ textAlign: "center" }}>
            {activeUser?.address || "Address"}
          </Text>
          <Text style={{ textAlign: "center" }}>Report File</Text>
        </View>

        <View style={styles.underline}></View>

        {/* Patient Details */}
        <View style={[styles.row, styles.section]}>
          <View style={{ width: "50%" }}>
            <Text style={styles.label}>Patient Name:</Text>
            <Text style={styles.text}>{slideDetail?.patientName || "N/A"}</Text>
          </View>
          <View style={{ width: "25%" }}>
            <Text style={styles.label}>Age:</Text>
            <Text style={styles.text}>{getAge(slideDetail?.dob)}</Text>
          </View>
          <View style={{ width: "25%" }}>
            <Text style={styles.label}>Sex:</Text>
            <Text style={styles.text}>
              {slideDetail?.patientGender || "N/A"}
            </Text>
          </View>
        </View>

        <View style={[styles.row, styles.section]}>
          <View style={{ width: "50%" }}>
            <Text style={styles.label}>Requesting Hospital:</Text>
            <Text style={styles.text}>
              {slideDetail?.referringInstitution || "-- --"}
            </Text>
          </View>
          <View style={{ width: "25%" }}>
            <Text style={styles.label}>Requesting Physician:</Text>
            <Text style={styles.text}>
              {slideDetail?.referringPhysician || "-- --"}
            </Text>
          </View>
          <View style={{ width: "25%" }}>
            <Text style={styles.label}>Date of Sample Collection:</Text>
            <Text style={styles.text}>
              {slideDetail?.dateOfSampleCollection
                ? moment(slideDetail?.dateOfSampleCollection).format(
                    "DD/MM/YYYY",
                  )
                : "-- --"}
            </Text>
          </View>
        </View>

        <View style={styles.underline}></View>

        {/* Report Sections */}
        <View style={styles.mainSection}>
          <View style={styles.textBlock}>
            <Text style={styles.label}>Clinical History:</Text>
            <Text style={styles.text}>
              {slideDetail?.clinicalHistory || "N/A"}
            </Text>
          </View>

          <View style={styles.textBlock}>
            <Text style={styles.label}>Nature of Specimen:</Text>
            <Text style={styles.text}>{slideDetail?.nature || "N/A"}</Text>
          </View>

          <View style={styles.textBlock}>
            <Text style={styles.label}>Macroscopic Description:</Text>
            <Text style={styles.text}>
              {slideDetail?.macroscopicDescription || "N/A"}
            </Text>
          </View>

          <View style={styles.textBlock}>
            <Text style={styles.label}>Microscopy:</Text>
            <Text style={styles.text}>{report?.microscopy || "N/A"}</Text>
          </View>

          <View style={styles.textBlock}>
            <Text style={styles.label}>Final Diagnosis:</Text>
            <Text style={styles.text}>{report?.diagnosis || "N/A"}</Text>
          </View>

          <View style={styles.textBlock}>
            <Text style={styles.label}>Comment (Recommendation):</Text>
            <Text style={styles.text}>{report?.recommendation || "N/A"}</Text>
          </View>
        </View>

        {/* Signature */}
        <View style={[styles.row, { marginTop: 20 }]}>
          <View style={{ width: "70%" }}>
            <Text style={styles.label}>
              Pathology Consultant’s Name & Signature
            </Text>

            <Text>{`Dr. ${activeUser?.firstName || ""} ${activeUser?.lastName || ""}`}</Text>
          </View>
          <View style={{ width: "30%" }}>
            <Text style={styles.label}>Date </Text>
            <Text>
              {slideDetail?.reportedAt
                ? moment(slideDetail?.reportedAt).format("DD/MM/YYYY")
                : ""}
            </Text>
          </View>
        </View>
      </Page>
    </Document>
  );
};

const ViewerReportDocument = ({
  activeUser,
  slideDetail,
  report,
}: {
  activeUser: User;
  slideDetail: any;
  report: { diagnosis: string; microscopy: string; recommendation: string };
}) => {
  const getAge = (birthDate: Date) => {
    return birthDate
      ? moment().diff(moment(birthDate, "YYYYMMDD"), "years")
      : "NA";
  };

  return (
    <Document title="Report Preview">
      <Page size="A4" style={styles.page}>
        {/* Institution Header */}
        <View style={styles.section}>
          <Text
            style={{ textAlign: "center", fontSize: 14, fontWeight: "bold" }}
          >
            {`${activeUser?.firstName || ""} ${activeUser?.lastName || " Report"}`}
          </Text>
          <Text style={{ textAlign: "center" }}>
            {activeUser?.address || "Address"}
          </Text>
          <Text style={{ textAlign: "center" }}>Report File</Text>
        </View>

        <View style={styles.underline}></View>

        {/* Patient Details */}
        <View style={[styles.row, styles.section]}>
          <View style={{ width: "50%" }}>
            <Text style={styles.label}>Patient Name:</Text>
            <Text style={styles.text}>{slideDetail?.patientName || "N/A"}</Text>
          </View>
          <View style={{ width: "25%" }}>
            <Text style={styles.label}>Age:</Text>
            <Text style={styles.text}>{getAge(slideDetail?.dob)}</Text>
          </View>
          <View style={{ width: "25%" }}>
            <Text style={styles.label}>Sex:</Text>
            <Text style={styles.text}>
              {slideDetail?.patientGender || "N/A"}
            </Text>
          </View>
        </View>

        <View style={[styles.row, styles.section]}>
          <View style={{ width: "50%" }}>
            <Text style={styles.label}>Requesting Hospital:</Text>
            <Text style={styles.text}>
              {slideDetail?.referringInstitution || "-- --"}
            </Text>
          </View>
          <View style={{ width: "25%" }}>
            <Text style={styles.label}>Requesting Physician:</Text>
            <Text style={styles.text}>
              {slideDetail?.referringPhysician || "-- --"}
            </Text>
          </View>
          <View style={{ width: "25%" }}>
            <Text style={styles.label}>Date of Sample Collection:</Text>
            <Text style={styles.text}>
              {slideDetail?.dateOfSampleCollection
                ? moment(slideDetail?.dateOfSampleCollection).format(
                    "DD/MM/YYYY",
                  )
                : "-- --"}
            </Text>
          </View>
        </View>

        <View style={styles.underline}></View>

        {/* Report Sections */}
        <View style={styles.mainSection}>
          <View style={styles.textBlock}>
            <Text style={styles.label}>Clinical History:</Text>
            <Text style={styles.text}>
              {slideDetail?.clinicalHistory || "N/A"}
            </Text>
          </View>

          <View style={styles.textBlock}>
            <Text style={styles.label}>Nature of Specimen:</Text>
            <Text style={styles.text}>{slideDetail?.nature || "N/A"}</Text>
          </View>

          <View style={styles.textBlock}>
            <Text style={styles.label}>Macroscopic Description:</Text>
            <Text style={styles.text}>
              {slideDetail?.macroscopicDescription || "N/A"}
            </Text>
          </View>

          <View style={styles.textBlock}>
            <Text style={styles.label}>Microscopy:</Text>
            <Text style={styles.text}>{report?.microscopy || "N/A"}</Text>
          </View>

          <View style={styles.textBlock}>
            <Text style={styles.label}>Final Diagnosis:</Text>
            <Text style={styles.text}>{report?.diagnosis || "N/A"}</Text>
          </View>

          <View style={styles.textBlock}>
            <Text style={styles.label}>Comment (Recommendation):</Text>
            <Text style={styles.text}>{report?.recommendation || "N/A"}</Text>
          </View>
        </View>

        {/* Signature */}
        <View style={[styles.row, { marginTop: 20 }]}>
          <View style={{ width: "70%" }}>
            <Text style={styles.label}>
              Pathology Consultant’s Name & Signature
            </Text>

            <Text>{`Dr. ${activeUser?.firstName || ""} ${activeUser?.lastName || ""}`}</Text>
          </View>
          <View style={{ width: "30%" }}>
            <Text style={styles.label}>Date </Text>
            <Text>
              {slideDetail?.reportedAt
                ? moment(slideDetail?.reportedAt).format("DD/MM/YYYY")
                : ""}
            </Text>
          </View>
        </View>
      </Page>
    </Document>
  );
};

const generateReportWithPdfLib = async ({
  activeUser,
  slideDetail,
  report,
}: {
  activeUser: any;
  slideDetail: any;
  report: {
    diagnosis: string;
    microscopy: string;
    recommendation: string;
  };
}) => {
  const dimensions = extractDimensions(
    activeUser?.pathologist?.letterHead as string,
  );

  const topMargin = dimensions?.top ?? 100;
  const bottomMargin = dimensions?.bottom ?? 100;

  const pdfDoc = await PDFDocument.create();

  const page = pdfDoc.addPage([595.28, 841.89]); // A4 size in points (width x height)
  const { width, height } = page.getSize();

  // Load and embed the letterhead background image
  const letterheadUrl = activeUser?.pathologist?.letterHead;

  const proxyUrl = `/api/proxy-image?url=${encodeURIComponent(letterheadUrl)}`;

  const response = await fetch(proxyUrl);
  const contentType = response.headers.get("Content-Type");
  const letterheadBytes = await response.arrayBuffer();

  let image;
  if (contentType === "image/png") {
    image = await pdfDoc.embedPng(letterheadBytes);
  } else if (contentType === "image/jpeg" || contentType === "image/jpg") {
    image = await pdfDoc.embedJpg(letterheadBytes);
  } else {
    throw new Error(`Unsupported image type: ${contentType}`);
  }

  page.drawImage(image, {
    x: 0,
    y: 0,
    width,
    height,
  });

  const font = await pdfDoc.embedFont(StandardFonts.Helvetica);
  const labelFontSize = 10;
  const textFontSize = 12;

  let y = height - topMargin; // Start below header

  const drawLabelAndText = (
    label: string,
    text: string,
    x: number,
    y: number,
  ) => {
    page.drawText(label, {
      x,
      y,
      size: labelFontSize,
      font,
      color: rgb(0.3, 0.3, 0.3),
    });
    page.drawText(text || "", {
      x,
      y: y - 14,
      size: textFontSize,
      font,
      color: rgb(0, 0, 0),
    });
  };

  const getAge = (dob: string) => {
    if (!dob) return "N/A";
    const dobDate = moment(dob, "YYYYMMDD");
    return moment().diff(dobDate, "years").toString();
  };

  page.drawLine({
    start: { x: 30, y: y + 20 },
    end: { x: width - 30, y: y + 20 },
    thickness: 1,
    color: rgb(0, 0, 0),
  });

  // Patient Details
  drawLabelAndText("Patient Name:", slideDetail?.patientName, 30, y);
  drawLabelAndText("Age:", getAge(slideDetail?.dob), 250, y);
  drawLabelAndText("Sex:", slideDetail?.patientGender, 400, y);

  y -= 40;

  // Request Info
  drawLabelAndText(
    "Requesting Hospital:",
    slideDetail?.referringInstitution || "-- --",
    30,
    y,
  );
  drawLabelAndText(
    "Requesting Physician:",
    slideDetail?.referringPhysician || "-- --",
    250,
    y,
  );
  drawLabelAndText(
    "Date of Sample Collection:",
    slideDetail?.dateOfSampleCollection
      ? moment(slideDetail?.dateOfSampleCollection).format("DD/MM/YYYY")
      : "-- --",
    400,
    y,
  );

  page.drawLine({
    start: { x: 30, y: y - 30 },
    end: { x: width - 30, y: y - 30 },
    thickness: 1,
    color: rgb(0, 0, 0),
  });

  y -= 60;

  const wrapText = (
    text: string,
    maxWidth: number,
    font: any,
    fontSize: number,
  ): string[] => {
    // Clean the text: remove carriage returns
    const cleanText = text.replace(/\r/g, "").replace(/\n/g, " ");
    const words = cleanText.split(" ");
    const lines: string[] = [];
    let currentLine = "";

    for (const word of words) {
      const testLine = currentLine ? currentLine + " " + word : word;
      const width = font.widthOfTextAtSize(testLine, fontSize);
      if (width <= maxWidth) {
        currentLine = testLine;
      } else {
        lines.push(currentLine);
        currentLine = word;
      }
    }

    if (currentLine) lines.push(currentLine);
    return lines;
  };

  const multilineText = (label: string, value: string, minHeight = 60) => {
    drawLabelAndText(label, "", 30, y);

    const lines = wrapText(value || "N/A", 500, font, textFontSize);

    lines.forEach((line: string, i: number) => {
      page.drawText(line, {
        x: 30,
        y: y - 15 - i * 14,
        size: textFontSize,
        font,
        color: rgb(0, 0, 0),
      });
    });

    // Calculate height used by content
    const contentHeight = 20 + lines.length * 14;

    // Enforce minimum section height
    const sectionHeight = Math.max(contentHeight, minHeight);

    // Prevent drawing below bottom margin
    if (y - sectionHeight < bottomMargin) {
      console.warn(
        `Cannot draw "${label}" section — would go below bottom margin.`,
      );
      return; // or handle a page break here in the future
    }

    y -= sectionHeight;
  };

  // Sections
  multilineText("Clinical History:", slideDetail?.clinicalHistory);
  multilineText("Nature of Specimen:", slideDetail?.nature);
  multilineText(
    "Macroscopic Description:",
    slideDetail?.macroscopicDescription,
  );
  multilineText("Microscopy:", report?.microscopy);
  multilineText("Final Diagnosis:", report?.diagnosis);
  multilineText("Comment (Recommendation):", report?.recommendation);

  // Signature
  y -= 30;
  drawLabelAndText("Pathology Consultant’s Name & Signature", "", 30, y);
  drawLabelAndText(
    `Dr. ${activeUser?.firstName || ""} ${activeUser?.lastName || ""}`,
    "",
    30,
    y - 12,
  );
  drawLabelAndText(
    "Date:",
    moment(slideDetail?.reportedAt).format("DD/MM/YYYY"),
    400,
    y,
  );

  // Serialize PDF and download
  const pdfBytes = await pdfDoc.save();
  const blob = new Blob([pdfBytes], { type: "application/pdf" });
  // saveAs(blob, "report.pdf");

  const url = URL.createObjectURL(blob);
  const a = document.createElement("a");
  a.href = url;
  a.download = `${slideDetail.patientName}-report.pdf`;
  document.body.appendChild(a);
  a.click();
  document.body.removeChild(a);
};

export const downloadReport = async ({
  activeUser,
  slideDetail,
  report,
}: {
  activeUser: User;
  slideDetail: any;
  report: { diagnosis: string; microscopy: string; recommendation: string };
}) => {
  try {
    const blob = await pdf(
      <ReportDocument
        activeUser={activeUser}
        slideDetail={slideDetail}
        report={report}
      />,
    ).toBlob();
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = `${slideDetail.patientName}-report.pdf`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
  } catch (error) {
    console.error("Error generating PDF", error);
  }
};
export const onDownloadReportFromViewer = async ({
  activeUser,
  slideDetail,
  report,
}: {
  activeUser: User;
  slideDetail: any;
  report: { diagnosis: string; microscopy: string; recommendation: string };
}) => {
  try {
    const blob = await pdf(
      <ViewerReportDocument
        activeUser={activeUser}
        slideDetail={slideDetail}
        report={report}
      />,
    ).toBlob();
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = `${slideDetail.patientName}-report.pdf`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
  } catch (error) {
    console.error("Error generating PDF", error);
  }
};
export const onDownloadReportWithoutHeader = async ({
  activeUser,
  slideDetail,
  report,
}: {
  activeUser: User;
  slideDetail: any;
  report: { diagnosis: string; microscopy: string; recommendation: string };
}) => {
  generateReportWithPdfLib({
    activeUser: activeUser,
    slideDetail,
    report,
  });
  // try {
  //   const blob = await pdf(
  //     <ReportDocumentWithoutHeader
  //       activeUser={activeUser}
  //       activeInstitution={activeInstitution}
  //       slideDetail={slideDetail}
  //       report={report}
  //     />,
  //   ).toBlob();
  //   const url = URL.createObjectURL(blob);
  //   const a = document.createElement("a");
  //   a.href = url;
  //   a.download = `${slideDetail.patientName}-report.pdf`;
  //   document.body.appendChild(a);
  //   a.click();
  //   document.body.removeChild(a);
  // } catch (error) {
  //   console.error("Error generating PDF", error);
  // }
};

export const generateReport = async ({
  activeUser,

  slideDetail,
  report,
}: {
  activeUser: User;
  slideDetail: any;
  report: { diagnosis: string; microscopy: string; recommendation: string };
}) => {
  try {
    const blob = await pdf(
      <ReportDocument
        activeUser={activeUser}
        slideDetail={slideDetail}
        report={report}
      />,
    ).toBlob();
    return blob;
  } catch (error) {
    console.error("Error generating PDF", error);
  }
};
