import React from "react";
import {
  Chart as ChartJS,
  ArcElement,
  CategoryScale,
  Filler,
  Legend,
  LinearScale,
  LineElement,
  PointElement,
  Title,
  Tooltip,
} from "chart.js";
import { Line } from "react-chartjs-2";
import { modalityColor } from "@/utils/utilities/constant";

ChartJS.register(
  ArcElement,
  CategoryScale,
  Filler,
  Legend,
  LinearScale,
  LineElement,
  PointElement,
  Title,
  Tooltip,
);

interface DistributionItem {
  period: string;
  ES?: number;
  GM?: number;
  OP?: number;
  SM?: number;
  XC?: number;
  Others?: number;
  [key: string]: string | number | undefined;
}

type ModalityKey = "ES" | "GM" | "OP" | "SM" | "XC" | "Others";

const LineChart = ({ data }: { data: DistributionItem[] }) => {
  const distribution = data?.map((d) => ({
    period: d.period,
    ES: d.ES || 0,
    GM: d.GM || 0,
    OP: d.OP || 0,
    SM: d.SM || 0,
    XC: d.XC || 0,
    Others: d.Others || 0,
  }));

  const periods: string[] = [
    ...new Set(distribution?.map((item) => item.period)),
  ].sort();

  const allKeys = distribution?.reduce((keys: ModalityKey[], item) => {
    Object.keys(item).forEach((key) => {
      if (
        key !== "period" &&
        key !== "undefined" &&
        !keys.includes(key as ModalityKey)
      ) {
        keys.push(key as ModalityKey);
      }
    });
    return keys;
  }, []);

  const chartData = {
    labels: periods,
    datasets: allKeys.map((key) => ({
      label: key,
      data: periods.map(
        (period) =>
          distribution.find((item) => item.period === period)?.[key] || 0,
      ),
      borderColor: modalityColor[key],
      pointStyle: "circle",
      pointBorderColor: "transparent",
      pointBackgroundColor: modalityColor[key],
      hoverRadius: 6,
      pointRadius: 0,
      pointHoverRadius: 6,
      pointHitRadius: 30,
      pointHoverBorderWidth: 2,
    })),
  };

  return (
    <div className="rounded-[8px] border border-greyStroke bg-whiteShade p-4">
      <h3 className="mb-5 text-lg font-semibold text-grey md:text-xl">
        Studies Report
      </h3>
      <Line
        data={chartData}
        options={{
          responsive: true,
          plugins: {
            legend: { position: "bottom" },
            tooltip: {
              yAlign: "bottom",
              xAlign: "right",
              mode: "index",
              intersect: false,
              backgroundColor: "#ffffff",
              //   boxShadow: '0 8px 5px #ADB5D915',
              position: "nearest",
              titleColor: "#ADB5D9",
              titleFont: {
                size: 12,
                style: "normal",
                family: "'Jost', sans-serif",
              },
              bodyColor: "#525768",
              bodyFont: {
                size: 11,
                style: "normal",
                family: "'Jost', sans-serif",
              },
              borderColor: "#F1F2F6",
              borderWidth: 1,
              bodySpacing: 10,
              padding: {
                x: 10,
                y: 8,
              },
              callbacks: {
                title: () => "",
                label: (context) => {
                  const { formattedValue, dataset } = context;
                  return `${dataset.label}: ${formattedValue}`;
                },
              },
            },
          },
          scales: {
            x: {
              grid: {
                display: false,
              },
            },
            y: {
              grid: {
                // borderDash: [5, 5],
              },
            },
          },
        }}
      />
    </div>
  );
};

export default LineChart;
