import { setSharedSlides, setSlides } from "@/redux/data/dataSlice";
import { AppDispatch, RootState } from "@/redux/store";
import slideService from "@/services/slide.service";
import { useCallback, useEffect, useMemo, useRef, useState } from "react";
import toast from "react-hot-toast";
import { useDispatch, useSelector } from "react-redux";

interface SearchState {
  uploaded: {
    placeholder: string;
    searchValue: string;
    fields: string[];
  };
  shared: {
    placeholder: string;
    searchValue: string;
    fields: string[];
  };
}

interface SearchParams {
  searchValue: string;
  fields: string[];
}

const useViewStudies = () => {
  const reduxDispatch = useDispatch<AppDispatch>();
  const timeoutRef = useRef<NodeJS.Timeout | null>(null);
  const [loadingUploaded, setLoadingUploaded] = useState<boolean>(false);
  const [loadingShared, setLoadingShared] = useState<boolean>(false);

  const { activeInstitution, activeUser } = useSelector(
    (state: RootState) => state.user,
  );
  const { slides, sharedSlides } = useSelector(
    (state: RootState) => state.data,
  );

  const [search, setSearch] = useState<SearchState>({
    uploaded: {
      placeholder: "Search by Name...",
      searchValue: "",
      fields: ["patientName", "modality"],
    },
    shared: {
      placeholder: "Search by Name...",
      searchValue: "",
      fields: ["patientName", "modality"],
    },
  });

  const isPathologist = useMemo(() => {
    return !!activeUser?.pathologist;
  }, [activeUser?.pathologist]);

  // Create ref for active institution
  const activeInstitutionRef = useRef(activeInstitution);

  // Keep ref updated
  useEffect(() => {
    activeInstitutionRef.current = activeInstitution;
  }, [activeInstitution]);

  useEffect(() => {
    return () => {
      if (timeoutRef.current) {
        clearTimeout(timeoutRef.current);
      }
    };
  }, []);

  const getSlides = useCallback(
    (searchParams: SearchParams) => {
      const currentInstitution = activeInstitutionRef.current;

      if (!currentInstitution?.id) {
        toast.error("No active institution selected");
        return;
      }

      if (!slides.length) setLoadingUploaded(true);
      if (!!searchParams.searchValue) setLoadingUploaded(true);

      slideService
        .getInstitutionSlides({
          institutionId: currentInstitution?.id,
          ...(searchParams && {
            search: searchParams.searchValue || "",
            searchFields: searchParams.fields,
          }),
        })
        .then((res) => {
          reduxDispatch(setSlides(res?.data as any[]));
        })
        .catch((error) => {
          console.log(error);
          toast.error(error?.message || error?.errors);
          setLoadingUploaded(false);
        })
        .finally(() => setLoadingUploaded(false));
    },
    [reduxDispatch, slides.length],
  );

  const getSharedSlides = useCallback(
    (searchParams: SearchParams) => {
      if (!isPathologist) return;

      const currentInstitution = activeInstitutionRef.current;

      if (!currentInstitution?.id) {
        toast.error("No active institution selected");
        return;
      }

      if (!sharedSlides.length) setLoadingShared(true);
      if (!!searchParams.searchValue) setLoadingShared(true);

      slideService
        .getInstitutionSharedSlides({
          institutionId: currentInstitution?.id,
          ...(searchParams && {
            search: searchParams.searchValue || "",
            searchFields: searchParams.fields,
          }),
        })
        .then((res) => {
          reduxDispatch(setSharedSlides(res?.data as any[]));
        })
        .catch((error) => {
          console.log(error);
          toast.error(error?.message || error?.errors);
          setLoadingShared(false);
        })
        .finally(() => setLoadingShared(false));
    },
    [reduxDispatch, sharedSlides.length, isPathologist],
  );

  const handleSearch = (
    e: React.ChangeEvent<HTMLInputElement>,
    type: keyof SearchState,
  ) => {
    const value = e.target.value;

    setSearch((prev) => {
      const updatedSearch = {
        ...prev,
        [type]: {
          ...prev[type],
          searchValue: value,
        },
      };

      // Clear existing timeout
      if (timeoutRef.current) {
        clearTimeout(timeoutRef.current);
      }

      // Set new debounced search
      timeoutRef.current = setTimeout(() => {
        if (type === "uploaded") {
          getSlides(updatedSearch.uploaded);
        }
      }, 500); // Optimal debounce time

      return updatedSearch;
    });
  };
  return {
    getSlides,
    loadingUploaded,
    search,
    handleSearch,
    getSharedSlides,
    loadingShared,
  };
};

export default useViewStudies;
