"use client";

import { ActionDropdown } from "@/components/Dropdowns/ActionDropdown";
import { ColumnConfig, generateColumns } from "@/components/Table/column";
import { DataTable } from "@/components/Table/DataTable";
import { Study } from "@/interfaces";
import { setSlideDetail } from "@/redux/data/dataSlice";
import { AppDispatch, RootState } from "@/redux/store";
import moment from "moment";
import { useRouter } from "next/navigation";
import React, { useCallback, useEffect, useState } from "react";
import { BsThreeDotsVertical } from "react-icons/bs";
import { MdRemoveRedEye } from "react-icons/md";
import { useDispatch, useSelector } from "react-redux";

interface SearchState {
  shared: {
    placeholder: string;
    searchValue: string;
    fields: string[];
  };
}

const Shared = ({
  isLoading,
  handleSearch,
  search,
}: {
  isLoading: boolean;
  handleSearch: (
    e: React.ChangeEvent<HTMLInputElement>,
    type: keyof SearchState,
  ) => void;
  search: {
    placeholder: string;
    searchValue: string;
    fields: string[];
  };
}) => {
  const router = useRouter();
  const reduxDispatch = useDispatch<AppDispatch>();

  const { sharedSlides } = useSelector((state: RootState) => state.data);
  const { activeUser, activeInstitution } = useSelector(
    (state: RootState) => state.user,
  );

  const studiesColumn: ColumnConfig<Study>[] = [
    {
      key: "patientName",
      header: "Patient Name",
    },
    {
      key: "modality",
      header: "Modality",
    },
    {
      key: "uploadProgress",
      header: "Uploaded",
    },
    {
      key: "createdAt",
      header: "Upload Date",
      cell: ({ row }) => {
        return (
          <div>{moment(row.original.createdAt).format("DD MMM YYYY")}</div>
        );
      },
    },
    {
      key: "status",
      header: "Status",
      cell: ({ row }) => {
        switch (row.original.status) {
          case "not_assigned":
            return (
              <div className="capitalize text-accentOrange">
                {row.original.status.replace("_", " ")}
              </div>
            );
          case "assigned":
            return (
              <div className="capitalize text-green-600">
                {row.original.status.replace("_", " ")}
              </div>
            );
          case "attended":
            return (
              <div className="capitalize text-primary">
                {row.original.status.replace("_", " ")}
              </div>
            );

          default:
            return (
              <div className="capitalize">
                {row.original.status.replace("_", " ")}
              </div>
            );
        }
      },
    },

    {
      key: "action",
      header: <p className="text-center">Action</p>,
      cell: ({ row }) => {
        return (
          <ActionDropdown
            theme="primary"
            align="end"
            className={{
              content: "w-fit bg-greyStroke",
              item: "hover:bg-white",
            }}
            trigger={
              <div className="flex justify-center">
                <BsThreeDotsVertical />
              </div>
            }
            options={[
              {
                label: <div>View Study</div>,
                onClick: () => {
                  reduxDispatch(setSlideDetail({}));
                  router.push(`/institution/view-studies/${row.original.id}`);
                },
                icon: <MdRemoveRedEye />,
              },
            ].filter(Boolean)}
          />
        );
      },
    },
  ];

  const columns = generateColumns(studiesColumn, {
    select: true,
    serialNumber: true,
  });

  const [filteredData, setFilteredData] = useState<Study[]>(sharedSlides);

  useEffect(() => {
    setFilteredData(sharedSlides);
  }, [sharedSlides]);

  const handleChange = useCallback((e: React.ChangeEvent<HTMLInputElement>) => {
    handleSearch(e, "shared");
  }, []);

  return (
    <React.Fragment>
      <DataTable
        theme="primary"
        data={filteredData}
        columns={columns}
        stickLeft={[2]}
        search={{ ...search, handleChange }}
        Loader={{
          isLoading: isLoading,
          loaderText: "Loading shared slides...",
        }}
        onRowClick={(data) => {
          reduxDispatch(setSlideDetail({}));
          router.push(`/institution/view-studies/${data?.id}`);
        }}
      />
    </React.Fragment>
  );
};

export default Shared;
