import { CaMicroscopeDataPayload, modalityTypes } from "@/interfaces";
import { FileStatusEnum, UploadingFile } from "@/interfaces/file";
import { RootState } from "@/redux/store";
import camicroscopeService from "@/services/camicroscope.service";
import slideService from "@/services/slide.service";
import { readChunkAsString } from "@/utils/utilities/file";
import { useRouter } from "next/navigation";
import { useEffect, useRef, useState } from "react";
import toast from "react-hot-toast";
import { useSelector } from "react-redux";

const chunkSize = 5 * 1024 * 1024;

interface FormValues {
  slideName: string;
  patientName: string;
  modality: modalityTypes | string;
  comment: string;
}

export const useCaMicroscopeUpload = () => {
  const router = useRouter();

  const [file, setFile] = useState<File | null>(null);

  const [isExtracting, setIsExtracting] = useState<boolean>(false);

  const [uploaded, setUploaded] = useState<boolean>(false);

  const [isUploading, setIsUploading] = useState<boolean>(false);

  const [uploadProgress, setUploadProgress] = useState<number>(0);

  const [isLoadingReal, setIsLoadingReal] = useState<boolean>(false);

  const { activeInstitution } = useSelector((state: RootState) => state.user);

  const initialValues: FormValues = {
    slideName: "",
    patientName: "",
    modality: "",
    comment: "",
  };

  const [formValues, setFormValues] = useState<FormValues>(initialValues);

  const formValuesRef = useRef(formValues);

  useEffect(() => {
    formValuesRef.current = formValues;
  }, [formValues]);

  const onSelectFIle = async (file: File) => {
    try {
      setIsExtracting(true);
      // console.log(file);
      setFile(file);
      setUploaded(true);
      setIsExtracting(false);
    } catch (error) {
      console.log(error);
    }
  };
  const onRemoveFile = async () => {
    try {
      setIsExtracting(false);
      setFile(file);
      setUploaded(false);
      const inputElement =
        document.querySelector<HTMLInputElement>("#input-field");
      if (inputElement) {
        inputElement.value = ""; // Clears the selected file
      }
    } catch (error) {
      console.log(error);
    }
  };

  // console.log("form", formValues);

  const startUpload = async (file: File) => {
    camicroscopeService
      .startUpload({ filename: file.name })
      .then(async (data) => {
        // console.log("--upload-request", data);
        const uploadingFile: UploadingFile = {
          file,
          fileSize: file.size,
          status: FileStatusEnum.UPLOADING,
          uploadingFileId: data.upload_token as string,
        };
        await startUploadInChunks(uploadingFile);
      })
      .catch((error) => {
        console.log(error);
        toast.error(error?.message);
        // console.log("oh shit");
        setIsUploading(false);
      })
      .finally();
  };

  const startUploadInChunks = async (uploadingFile: UploadingFile) => {
    const { file, uploadingFileId } = uploadingFile;
    // const totalChunks = Math.ceil(file.size / chunkSize);
    // const chunkProgress = 100 / totalChunks;

    setUploadProgress(0);

    let chunkNumber = 1;
    let start = 0;
    let end = start + chunkSize;

    const uploadNextChunk = async () => {
      if (start <= file.size) {
        const chunkedFile: any = file.slice(start, end);
        const chunk = await readChunkAsString(chunkedFile);

        const payload = {
          chunkSize,
          data: chunk,
          offset: chunkNumber * chunkSize,
        };

        try {
          camicroscopeService
            .uploadChunk(uploadingFileId, payload)
            .then((data) => {
              chunkNumber++;
              start = end;
              end = start + chunkSize;

              const progress = Math.min(
                100,
                Number(
                  (((payload.offset + chunkSize) / file.size) * 100).toFixed(0),
                ),
              );

              setUploadProgress(progress);

              if (end >= file.size) {
                uploadLastChunk(uploadingFile);
              } else {
                uploadNextChunk();
              }
            })
            .catch((error) => {
              console.log(error, "<<-== error uploading chunks");
              toast.error(error?.message);
            });
        } catch (error) {
          console.log(error, "<<-== error 2 uploading chunks");
          setIsUploading(false);
        }
      }
    };
    uploadNextChunk();
  };

  const uploadLastChunk = (uploadingFile: UploadingFile) => {
    const { uploadingFileId, file } = uploadingFile;
    camicroscopeService
      .finishUpload(uploadingFileId, { filename: file.name })
      .then((data) => {
        getUploadedMetadata(`${data.filename}`);
      })
      .catch((error) => {
        console.log(error, "<<-== error on finish upload");
        toast.error(error?.message);
        setIsUploading(false);
      });
  };

  const getUploadedMetadata = (fileName: string) => {
    camicroscopeService
      .getUploadedMetadata(fileName)
      .then((data) => {
        const currentFormValues = formValuesRef.current;

        finalUploadProxy({
          width: data?.width as string,
          height: data?.height as string,
          "mpp-x": data?.["mpp-x"] as string,
          "mpp-y": data?.["mpp-y"] as string,
          vendor: data?.vendor as string,
          level_count: data?.level_count as number,
          objective: data?.objective as number,
          study: data?.study as string,
          specimen: data?.specimen as string,
          md5: (data?.md5 as string) || (data?.md5sum as string),
          location: data?.location as string,
          filepath: data?.filepath as string,
          upload_date: data?.upload_date as string,
          name: currentFormValues.slideName || "",
          comment:
            `${data?.comment || ""} ${currentFormValues.comment || ""}`.trim(),
        });
      })
      .catch((error) => {
        console.log(error, "<<-== error on getmetadata");
        toast.error(error?.message);
        setIsUploading(false);
      })
      .finally(() => {});
  };

  const finalUploadProxy = async (body: CaMicroscopeDataPayload) => {
    const response = await fetch("/api/proxy-upload", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify(body),
    });

    if (!response.ok) {
      const errorData = await response.json();
      throw new Error(errorData.error || "Proxy upload failed");
    }

    const data = await response.json();
    // console.log("response-data", data);
    const dbBody = data.ops[0];
    UploadToXolaniDB(dbBody);
  };

  const UploadToXolaniDB = async (body: CaMicroscopeDataPayload) => {
    const currentFormValues = formValuesRef.current;

    if (activeInstitution && file) {
      slideService
        .uploadSlideInstitution({
          institutionId: activeInstitution?.id,
          body: {
            slideImageHost: "camicroscope",
            patientName: currentFormValues.patientName,
            name: currentFormValues.slideName,
            modality: currentFormValues.modality,
            comment: body.comment,
            fileSize: file?.size || 0,
            caMicroscopeData: {
              width: body.width,
              height: body.height,
              mppX: body["mpp-x"] as string,
              mppY: body["mpp-y"] as string,
              vendor: body.vendor,
              levelCount: body.level_count,
              objective: body.objective,
              comment: body.comment,
              study: body.study,
              specimen: body.specimen,
              md5: body.md5,
              location: body.location, //Required
              filepath: body.filepath, //Required
              uploadDate: new Date().toLocaleString("en-US"),
              name: body.name, //Required: slide name
              camSlideId: body?._id as string, //Required: The unique ID assigned to the slide after successful upload
              fileName: file?.name, //Required
              fileToken: body?._id as string,
            },
          },
        })
        .then((response) => {
          toast.success(response?.message);
          setFormValues(initialValues);
          router.push("/institution/view-studies");
        })
        .catch((error) => {
          console.log(error, "<<-== error on SlideUploadToDB");
          toast.error(error?.message);
          setIsUploading(false);
        })
        .finally(() => setIsUploading(false));
    }
  };

  const onCustomUpload = async (file: File) => {
    try {
      setIsUploading(true);

      const response = await camicroscopeService.getUploadedMetadata(file.name);

      if (response.error === "No such file") {
        await startUpload(file);
        return;
      }

      getUploadedMetadata(file.name);
    } catch (error) {
      console.log("Error uploading file:", error);

      setIsUploading(false);
    }
  };

  const submitFileForUpload = () => {
    if (!file) {
      return toast.error("Please upload a valid  file.");
    }

    onCustomUpload(file);
  };

  return {
    file,
    isExtracting,
    uploaded,
    handleFileUpload: onSelectFIle,
    handleSubmitStudy: submitFileForUpload,
    isUploading,
    uploadProgress,
    formValues,
    setFormValues,
    onRemoveFile,
    isLoadingReal,
    setIsLoadingReal,
  };
};
