"use client";

import { getInitials } from "@/utils/utilities";
import React, { useEffect, useMemo, useRef, useState } from "react";
import {
  MdOutlineDelete,
  MdOutlineEdit,
  MdOutlinePerson,
  MdOutlineVpnKey,
} from "react-icons/md";
import { AiOutlineControl } from "react-icons/ai";
import SettingItem from "./SettingItem";
import { RootState } from "@/redux/store";
import { useSelector } from "react-redux";
import { usePathname } from "next/navigation";
import { Button } from "@/components/Buttons";
import toast from "react-hot-toast";
import Image from "next/image";
import institutionService from "@/services/institution.service";
import useFetchData from "../../(hooks)/useFetchData";

const SettingMenu = () => {
  const pathname = usePathname();
  const fileInputRef = useRef<HTMLInputElement>(null);
  const [logo, setLogo] = useState<string | null>(null);
  const [logoPreview, setLogoPreview] = useState<string | null>(null);
  const [logoToUpload, setLogoToUpload] = useState<File | null>(null);
  const [isLoading, setIsLoading] = useState<boolean>(false);

  const { activeRole, activeUser, activeInstitution } = useSelector(
    (state: RootState) => state.user,
  );
  const { getUserDetails } = useFetchData();

  useEffect(() => {
    setLogo(activeInstitution?.logo as string | null);
  }, [activeInstitution]);

  const menuPath = pathname.split("/")[3];

  const getName = useMemo(() => {
    if (activeRole === "PATHOLOGIST") {
      return `${activeUser?.firstName} ${activeUser?.lastName}`;
    } else if (activeRole === "SUPERADMIN") {
      return activeInstitution?.institutionName;
    } else if (activeRole === "ADMIN") {
      return `${activeUser?.firstName} ${activeUser?.lastName}`;
    } else return "";
  }, [activeRole, activeUser, activeInstitution]);

  const handleFileUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    const allowedTypes = [
      "image/png",
      "image/jpeg",
      "image/jpg",
      "image/gif",
      "image/webp",
      "image/svg+xml",
    ];

    if (!allowedTypes.includes(file.type)) {
      toast.error("Only image files are allowed.");
      return;
    }

    const imageUrl = URL.createObjectURL(file);
    setLogoPreview(imageUrl);
    setLogoToUpload(file);
  };

  const onRemovePreview = () => {
    setLogoPreview(null);
    setLogoToUpload(null);
  };

  const handleUploadToCloud = () => {
    if (!logoToUpload || !activeInstitution) return;

    const formData = new FormData();

    setIsLoading(true);
    toast.loading("Uploading logo");

    formData.append("file", logoToUpload);

    institutionService
      .uploadLogo({
        institutionId: activeInstitution?.id,
        body: formData,
      })
      .then((res) => {
        toast.success(res?.message);
        getUserDetails();
        setLogoPreview(null);
        setLogoToUpload(null);
      })
      .catch((error) => {
        console.log(error);
        toast.error(error?.message || error?.errors);
      })
      .finally(() => setIsLoading(false));
  };

  const onDeleteFromCloud = () => {
    if (!activeInstitution) return;

    setIsLoading(true);
    toast.loading("Deleting logo");

    institutionService
      .removeLogo({
        institutionId: activeInstitution?.id,
      })
      .then((res) => {
        toast.success(res?.message);
        getUserDetails();
        setLogoPreview(null);
        setLogoToUpload(null);
      })
      .catch((error) => {
        console.log(error);
        toast.error(error?.message || error?.errors);
      })
      .finally(() => setIsLoading(false));
  };

  return (
    <div>
      <div className="mx-auto flex w-3/5 flex-col items-center justify-center bg-lightGreenLight p-5 py-8">
        {logoPreview ? (
          <Image
            className="size-[110px] overflow-hidden rounded-full object-cover"
            width={110}
            height={110}
            src={logoPreview}
            alt="logo preview"
          />
        ) : logo ? (
          <Image
            className="size-[110px] overflow-hidden rounded-full object-cover"
            width={110}
            height={110}
            src={logo}
            alt="logo"
          />
        ) : (
          <div className="flex size-[110px] items-center justify-center overflow-hidden rounded-full bg-primary object-cover text-5xl text-whiteShade shadow-lg">
            {getInitials(getName || "")}
          </div>
        )}

        <div className="mt-2 flex items-center gap-2 text-greySecondary">
          {/* Show "Edit Logo" button if no preview is active */}
          {!logoPreview && (
            <Button
              iconPrefix={<MdOutlineEdit size={20} />}
              variant="normal"
              onClick={() => fileInputRef.current?.click()}
              className="px-2 hover:bg-buttonGradient hover:text-white"
            >
              Logo
            </Button>
          )}

          {/* Show Upload button if preview is active */}
          {logoPreview && (
            <Button
              loading={isLoading}
              variant="normal"
              className="px-2 hover:bg-buttonGradient hover:text-white"
              onClick={handleUploadToCloud} // You should define this
            >
              Upload
            </Button>
          )}

          {/* Hidden file input */}
          <input
            type="file"
            ref={fileInputRef}
            className="hidden"
            accept="image/png, image/jpeg, image/jpg, image/gif, image/webp, image/svg+xml"
            onChange={handleFileUpload}
          />

          {/* Delete buttons */}
          {logoPreview ? (
            <Button
              variant="normal"
              onClick={onRemovePreview}
              className="px-2 hover:bg-accentDanger hover:text-white"
            >
              <MdOutlineDelete size={20} />
            </Button>
          ) : logo ? (
            <Button
              variant="normal"
              onClick={onDeleteFromCloud} // You should define this
              className="px-2 hover:bg-accentDanger hover:text-white"
            >
              <MdOutlineDelete size={20} />
            </Button>
          ) : null}
        </div>
      </div>

      <div className="mt-5 rounded-[8px] border border-greyStroke bg-whiteShade">
        <div className="border-b border-greyStroke px-10 py-8">
          <p className="text-2xl font-semibold text-grey">{getName}</p>
        </div>
        <nav>
          <ul className="space-y-2 px-6 py-8">
            <SettingItem
              label="Profile"
              to="/institution/settings/profile"
              icon={<MdOutlinePerson size={20} />}
              isActive={menuPath === "profile"}
            />
            <SettingItem
              label="Password"
              to="/institution/settings/password"
              icon={<MdOutlineVpnKey size={20} />}
              isActive={menuPath === "password"}
            />
            {activeRole === "SUPERADMIN" && (
              <SettingItem
                label="Controls"
                to="/institution/settings/controls"
                icon={<AiOutlineControl size={20} />}
                isActive={menuPath === "controls"}
              />
            )}
          </ul>
        </nav>
      </div>
    </div>
  );
};

export default SettingMenu;
