import { setPathologists } from "@/redux/data/dataSlice";
import { setPromptLoading, togglePopup } from "@/redux/global/globalSlice";
import { AppDispatch, RootState } from "@/redux/store";
import institutionService from "@/services/institution.service";
import { useRef, useState } from "react";
import toast from "react-hot-toast";
import { useDispatch, useSelector } from "react-redux";

interface SearchState {
  placeholder: string;
  searchValue: string;
  fields: string[];
}

interface SearchParams {
  searchValue: string;
  fields: string[];
}
const usePathologists = () => {
  const reduxDispatch = useDispatch<AppDispatch>();
  const timeoutRef = useRef<NodeJS.Timeout | null>(null);

  const [loadingPathologists, setLoadingPathologists] =
    useState<boolean>(false);

  const { activeInstitution } = useSelector((state: RootState) => state.user);
  const { pathologists } = useSelector((state: RootState) => state.data);

  const [search, setSearch] = useState<SearchState>({
    placeholder: "Search by Name...",
    searchValue: "",
    fields: ["firstName", "lastName", "email"],
  });

  const getPathologists = async (searchParams: SearchParams) => {
    if (activeInstitution) {
      if (!pathologists.length) setLoadingPathologists(true);
      if (!!searchParams.searchValue) setLoadingPathologists(true);

      institutionService
        .getInstitutionPathologists({
          institutionId: activeInstitution?.id,
          ...(searchParams && {
            search: searchParams.searchValue || "",
            searchFields: searchParams.fields,
          }),
        })
        .then((res) => {
          reduxDispatch(setPathologists(res?.data as any[]));
        })
        .catch((error) => {
          console.log(error);
          toast.error(error?.message || error?.errors);
          setLoadingPathologists(false);
        })
        .finally(() => setLoadingPathologists(false));
    }
  };

  const removePathologist = async (pathId: number) => {
    if (activeInstitution) {
      reduxDispatch(setPromptLoading(true));

      institutionService
        .removePathologist({
          institutionId: activeInstitution?.id,
          pathologistId: pathId,
        })
        .then((res) => {
          toast(res?.message || "Successfully removed");
          getPathologists(search);
          reduxDispatch(togglePopup({ key: "firstPopup", isOpen: false }));
        })
        .catch((error) => {
          console.log(error);
          toast.error(error?.message || error?.errors);
        })
        .finally(() => reduxDispatch(setPromptLoading(false)));
    }
  };

  const handleSearch = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value;

    setSearch((prev) => {
      const updatedSearch = {
        ...prev,
        searchValue: value,
      };

      // Clear existing timeout
      if (timeoutRef.current) {
        clearTimeout(timeoutRef.current);
      }

      // Set new debounced search
      timeoutRef.current = setTimeout(() => {
        getPathologists(updatedSearch);
      }, 500); // Optimal debounce time

      return updatedSearch;
    });
  };
  return {
    getPathologists,
    loadingPathologists,
    search,
    handleSearch,
    removePathologist,
  };
};

export default usePathologists;
