"use client";

import { Button } from "@/components/Buttons";
import { PhoneInput, TextInput } from "@/components/Inputs";
import { setDrawer, togglePopup } from "@/redux/global/globalSlice";
import { AppDispatch, RootState } from "@/redux/store";
import slideService from "@/services/slide.service";
import { MANAGE_SHARE_TO_EXTERNAL_SCHEMA } from "@/utils/validationSchema/institution.schema";
import { Form, Formik } from "formik";
import { useParams } from "next/navigation";
import React, { useState } from "react";
import { useDispatch, useSelector } from "react-redux";
import { generateReport } from "../view-studies/components/ViewReport";
import toast from "react-hot-toast";

interface FormValues {
  email: string;
  name: string;
  phone: string;
  validity: number | null | string;
  code: string;
}

const SendToExternalPathologist = ({
  body,
  reportId,
}: {
  body?: { diagnosis: string; microscopy: string; recommendation: string };
  reportId?: number;
}) => {
  const reduxDispatch = useDispatch<AppDispatch>();
  const { studyId } = useParams();

  const { activeInstitution, activeUser } = useSelector(
    (state: RootState) => state.user,
  );
  const { slideDetail } = useSelector((state: RootState) => state.data);

  const [isLoading, setIsLoading] = useState<boolean>(false);
  const [validityDisplay, setValidityDisplay] = useState<string>("");

  const initialValues: FormValues = {
    email: "",
    name: "",
    phone: "",
    validity: "",
    code: "",
  };

  const calculateValidityDisplay = (minutes: number): string => {
    const days = Math.floor(minutes / 1440);
    const remainingAfterDays = minutes % 1440;
    const hours = Math.floor(remainingAfterDays / 60);
    const mins = remainingAfterDays % 60;

    const parts = [];
    if (days > 0) parts.push(`${days} day${days !== 1 ? "s" : ""}`);
    if (hours > 0) parts.push(`${hours}Hr${hours !== 1 ? "s" : ""}`);
    if (mins > 0) parts.push(`${mins}min${mins !== 1 ? "s" : ""}`);

    return parts.length > 0
      ? `${parts.join(" ")} Valid Period`
      : "0min Valid Period";
  };

  const onSave = async (values: FormValues) => {
    if (activeInstitution && studyId && activeUser && body) {
      setIsLoading(true);

      const blob = await generateReport({
        activeUser,
        activeInstitution,
        slideDetail: slideDetail,
        report: body,
      });

      const formData = new FormData();

      const validityInSeconds =
        values?.validity && (values.validity as number) * 60;

      if (blob) {
        formData.append("file", blob, `${slideDetail.patientName}-report.pdf`);
      }

      formData.append("diagnosis", body.diagnosis);
      formData.append("microscopy", body.microscopy);
      formData.append("recommendations", body.recommendation);
      formData.append("impressions", "");
      formData.append("isFinal", JSON.stringify(true));
      formData.append("action", "CREATE_AND_SHARE_EXTERNALLY");
      formData.append("externalShareInfo[name]", String(values.name));
      formData.append("externalShareInfo[email]", String(values.email));
      formData.append("externalShareInfo[phoneNumber]", String(values.phone));
      formData.append(
        "externalShareInfo[validityPeriod]",
        String(validityInSeconds ?? ""),
      );
      formData.append("externalShareInfo[code]", String(values.code));

      slideService
        .createInstitutionSlideReport({
          institutionId: activeInstitution.id,
          slideId: studyId as string,
          body: formData,
        })
        .then(() => {
          reduxDispatch(togglePopup({ key: "firstPopup", isOpen: false }));
          reduxDispatch(setDrawer({ isOpen: false }));
          toast.success("Report Shared to External");
        })
        .catch((error) => {
          toast.error(error?.message || "Failed to send report");
        })
        .finally(() => setIsLoading(false));
    }
  };

  const onSend = async (values: FormValues) => {
    if (activeInstitution && studyId && activeUser && reportId) {
      setIsLoading(true);

      const validityInSeconds =
        values?.validity && (values.validity as number) * 60;

      slideService
        .shareReportExternal({
          institutionId: activeInstitution.id,
          reportId: Number(reportId),
          externalShareInfo: {
            name: values.name,
            email: values.email,
            phoneNumber: values.phone,
            validityPeriod: (validityInSeconds as number) ?? 0,
            code: values.code,
          },
        })
        .then(() => {
          reduxDispatch(togglePopup({ key: "firstPopup", isOpen: false }));
          toast.success("Report Shared to External");
        })
        .catch((error) => {
          toast.error(error?.message || "Failed to send report");
        })
        .finally(() => setIsLoading(false));
    }
  };

  return (
    <div className="flex h-[600px] w-[85vw] flex-col rounded-lg bg-white sm:w-[500px]">
      <div className="mb-3 px-3 pb-3 pt-8 text-center shadow-sm sm:px-8">
        <span className="text-xl font-semibold text-grey">
          Share Report to External Pathologist
        </span>
      </div>
      <div className="relative flex flex-1 flex-col overflow-auto px-3 pb-4 pt-2 sm:px-8">
        <p className="text-center text-sm font-medium text-greySecondary">
          Would you like to send your report to a pathologist for review?
        </p>

        <Formik
          initialValues={initialValues}
          validationSchema={MANAGE_SHARE_TO_EXTERNAL_SCHEMA}
          onSubmit={reportId ? onSend : onSave}
        >
          {({ setFieldValue }) => (
            <Form className="mt-4 space-y-1" id="sendToExternalPathologist">
              <TextInput
                name="name"
                type="text"
                label="Pathologist Name"
                disabled={isLoading}
              />

              <TextInput
                name="email"
                type="email"
                label="Email Address"
                disabled={isLoading}
              />
              <PhoneInput name="phone" label="Phone No." />
              <TextInput
                name="validity"
                type="text"
                label="Validity Period (in Minutes)"
                disabled={isLoading}
                onChange={(e) => {
                  const sanitizedValue = e.target.value.replace(/[^0-9]/g, "");
                  const numericValue =
                    sanitizedValue === "" ? null : parseInt(sanitizedValue, 10);
                  setFieldValue("validity", numericValue);

                  const minutes = numericValue || 0;
                  setValidityDisplay(calculateValidityDisplay(minutes));
                }}
              />
              {validityDisplay && (
                <div className="mt-1 text-sm text-blackMain">
                  {validityDisplay}
                </div>
              )}
              <TextInput
                name="code"
                type="text"
                label="Access Code"
                disabled={isLoading}
              />
            </Form>
          )}
        </Formik>
      </div>
      <div className="flex w-full justify-between border-t-2 border-greyStroke px-3 py-6 sm:px-8">
        <Button
          variant="gray"
          type="button"
          onClick={() =>
            reduxDispatch(togglePopup({ key: "firstPopup", isOpen: false }))
          }
        >
          Cancel
        </Button>
        <Button
          variant="primary"
          type="submit"
          form="sendToExternalPathologist"
          loading={isLoading}
        >
          Share
        </Button>
      </div>
    </div>
  );
};

export default SendToExternalPathologist;
