"use client";

import { Button } from "@/components/Buttons";
import { TextInput } from "@/components/Inputs";
import Loading from "@/components/Loading/Loading";
import { Checkbox } from "@/components/ui/checkbox";
import { togglePopup } from "@/redux/global/globalSlice";
import { AppDispatch, RootState } from "@/redux/store";
import slideService from "@/services/slide.service";
import { getInitials } from "@/utils/utilities";
import { Form, Formik } from "formik";
import { useParams } from "next/navigation";
import React, { useEffect, useState } from "react";
import toast from "react-hot-toast";
import { useDispatch, useSelector } from "react-redux";
import usePathologists from "../pathologists/hooks/usePathologists";

const SendForReview = ({
  body,
}: {
  body: { diagnosis: string; microscopy: string; recommendation: string };
}) => {
  const reduxDispatch = useDispatch<AppDispatch>();
  const [isLoading, setIsLoading] = useState<boolean>(false);
  const [selected, setSelected] = useState<string | null>(null);
  const { studyId } = useParams();

  const { pathologists } = useSelector((state: RootState) => state.data);
  const { activeInstitution } = useSelector((state: RootState) => state.user);
  const { getPathologists, loadingPathologists, search, handleSearch } =
    usePathologists();

  useEffect(() => {
    if (activeInstitution) {
      getPathologists(search);
    }
  }, [activeInstitution?.id]);

  const onSelect = (value: boolean, pathologistId: string) => {
    if (value) {
      setSelected(pathologistId);
    } else {
      setSelected(null);
    }
  };

  const onSave = () => {
    if (activeInstitution && studyId && selected) {
      setIsLoading(true);
      const formData = new FormData();

      formData.append("diagnosis", body.diagnosis);
      formData.append("microscopy", body.microscopy);
      formData.append("recommendations", body.recommendation);
      formData.append("pathologistId", JSON.stringify(Number(selected)));
      formData.append("impressions", "");
      formData.append("send", JSON.stringify(false));
      formData.append("isFinal", JSON.stringify(false));
      formData.append("action", "CREATE_AND_SHARE");

      slideService
        .createInstitutionSlideReport({
          institutionId: activeInstitution.id,
          slideId: studyId as string,
          body: formData,
        })
        .then(() => {
          reduxDispatch(togglePopup({ key: "firstPopup", isOpen: false }));
          toast.success("Report sent for review successfully");
        })
        .catch((error) => {
          toast.error(error?.message || "Failed to send report");
        })
        .finally(() => setIsLoading(false));
    }
  };

  return (
    <div className="flex h-[500px] w-[85vw] flex-col rounded-lg bg-white sm:w-[500px]">
      <div className="mb-3 px-3 pb-3 pt-8 text-center shadow-sm sm:px-8">
        <span className="text-xl font-semibold text-grey">
          Share Report For Review
        </span>
      </div>
      <div className="relative flex flex-1 flex-col overflow-auto px-3 pb-4 pt-0 sm:px-8">
        <p className="text-center text-sm font-medium text-greySecondary">
          Would you like to send your report to a pathologist for review?
        </p>

        {loadingPathologists ? (
          <div className="mb-0 flex flex-1 items-center justify-center">
            <Loading loaderText="Loading pathologists..." />
          </div>
        ) : pathologists.length > 0 ? (
          <>
            {isLoading && <Loading loaderText="Sending report..." />}
            <div className="pt-3">
              <Formik initialValues={{ search: "" }} onSubmit={() => {}}>
                {() => (
                  <Form>
                    <TextInput
                      name="search"
                      type="search"
                      onChange={(e) =>
                        handleSearch(e as React.ChangeEvent<HTMLInputElement>)
                      }
                      placeholder="Search pathologists..."
                    />
                  </Form>
                )}
              </Formik>
            </div>

            <ul className="mb-0 mt-4 h-[100%] overflow-y-auto rounded-[4px] border border-greyStroke px-4 py-2">
              <p className="text-[14px] font-medium text-greySecondary">
                Pathologist List
              </p>
              {pathologists
                .filter((pathologist) => pathologist.type === "CONSULTANT")
                .map((pathologist) => (
                  <li
                    key={pathologist.id}
                    className="my-3 flex flex-row items-center justify-between"
                  >
                    <div className="inline-flex items-center gap-2">
                      <span className="flex size-10 items-center justify-center rounded-full bg-primaryMain p-1 text-center text-[16px] font-medium text-whiteShade">
                        {getInitials(
                          `${pathologist.pathologist.user.firstName} ${pathologist.pathologist.user.lastName}`,
                        )}
                      </span>
                      <div className="flex flex-col justify-center gap-[-5px]">
                        <span className="font-medium capitalize text-grey">
                          {pathologist.pathologist.user.firstName}{" "}
                          {pathologist.pathologist.user.lastName}
                        </span>
                        <span className="text-sm font-normal text-greySecondary">
                          {pathologist.pathologist.user.email}
                        </span>
                      </div>
                    </div>
                    <div className="">
                      <Checkbox
                        onCheckedChange={(value: boolean) =>
                          onSelect(value, pathologist.pathologist.id)
                        }
                        checked={pathologist.pathologist.id === selected}
                      />
                    </div>
                  </li>
                ))}
            </ul>
          </>
        ) : (
          <div className="mb-0 flex flex-1 flex-col items-center justify-center overflow-y-auto pt-3">
            <p className="px-4 text-center text-sm text-greySecondary">
              No pathologists found. Add pathologists to your institution first.
            </p>
          </div>
        )}
      </div>
      <div className="flex w-full justify-between border-t-2 border-greyStroke px-3 py-6 sm:px-8">
        <Button
          variant="gray"
          type="button"
          onClick={() =>
            reduxDispatch(togglePopup({ key: "firstPopup", isOpen: false }))
          }
        >
          Cancel
        </Button>
        <Button
          variant="primary"
          type="button"
          loading={isLoading}
          disabled={!selected}
          onClick={onSave}
        >
          Send
        </Button>
      </div>
    </div>
  );
};

export default SendForReview;
