import { Button } from "@/components/Buttons";
import { SelectInput, TextInput } from "@/components/Inputs";
import { modalityTypes } from "@/interfaces";
import { togglePopup } from "@/redux/global/globalSlice";
import { AppDispatch, RootState } from "@/redux/store";
import { MANAGE_UPDATE_SLIDE_SCHEMA } from "@/utils/validationSchema/institution.schema";
import { Form, Formik } from "formik";
import React, { useState } from "react";
import { useDispatch, useSelector } from "react-redux";
import { modalityTypesArray } from "../constants";
import institutionService from "@/services/institution.service";
import toast from "react-hot-toast";
import useViewStudies from "../view-studies/hooks/useViewStudies";
import useSlideDetail from "../view-studies/[studyId]/hooks/useSlideDetail";

interface FormValues {
  patientName: string;
  modality: modalityTypes | string;
  comment: string;
  macroscopicDescription: string;
}
interface Slide {
  id: string | number;
  patientName?: string | null;
  modality?: string | null;
  comment?: string | null;
  macroscopicDescription?: string | null;
}

const ManageUpdateSlide = ({
  slide,
  isDetail,
}: {
  slide: Slide;
  isDetail: boolean;
}) => {
  const reduxDispatch = useDispatch<AppDispatch>();
  const [isLoading, setIsLoading] = useState<boolean>();

  const { activeInstitution } = useSelector((state: RootState) => state.user);

  const initialValues: FormValues = {
    patientName: slide?.patientName || "",
    modality: slide?.modality || "",
    macroscopicDescription: slide?.macroscopicDescription || "",
    comment: slide?.comment || "",
  };

  const { getSlides, search } = useViewStudies();
  const { getSlideDetail } = useSlideDetail();

  const onHandleSubmit = async (values: FormValues) => {
    if (activeInstitution) {
      setIsLoading(true);
      institutionService
        .updateSlideData({
          slideId: slide.id,
          institutionId: activeInstitution.id,
          ...values,
        })
        .then((res) => {
          toast.success(res.message);
          if (isDetail) {
            getSlideDetail();
          } else getSlides(search.uploaded);

          reduxDispatch(togglePopup({ key: "firstPopup", isOpen: false }));
        })
        .catch((error) => {
          console.log(error);
          toast.error(error?.message || error?.errors);
        })
        .finally(() => {
          setIsLoading(false);
        });
    }
  };

  return (
    <div className="flex h-[500px] w-[85vw] flex-col rounded-lg bg-white sm:w-[500px]">
      <div className="mb-3 px-3 pb-3 pt-8 text-center shadow-sm sm:px-8">
        <span className="text-xl font-semibold text-grey">
          Update Slide Data
        </span>
      </div>
      <div className="relative flex flex-1 flex-col overflow-auto px-3 pb-4 pt-2 sm:px-8">
        <Formik
          initialValues={initialValues}
          validationSchema={MANAGE_UPDATE_SLIDE_SCHEMA}
          onSubmit={onHandleSubmit}
        >
          {() => (
            <Form className="mt-4 space-y-1" id="manageUpdateSlide">
              <TextInput
                type="text"
                name="patientName"
                label="Patient Name"
                placeholder="Enter patient name"
                disabled={isLoading}
              />
              <SelectInput
                name="modality"
                options={modalityTypesArray}
                disabled={isLoading}
                type="primary"
                label="Select Modality"
                placeholder="Select Modality"
                className={{
                  content: "w-full",
                  trigger:
                    "h-[40px] w-full rounded-[4px] border-greyLine text-blackMain outline-none [&.placeholder]:!text-gray-500",
                }}
              />
              <TextInput
                type="textarea"
                name="macroscopicDescription"
                label="Macroscopic Description (Optional)"
                disabled={isLoading}
                rows={4}
                placeholder="Enter additional info"
              />
            </Form>
          )}
        </Formik>
      </div>
      <div className="flex w-full justify-between border-t-2 border-greyStroke px-3 py-6 sm:px-8">
        <Button
          variant="gray"
          type="button"
          onClick={() =>
            reduxDispatch(togglePopup({ key: "firstPopup", isOpen: false }))
          }
        >
          Cancel
        </Button>
        <Button
          variant="primary"
          type="submit"
          form="manageUpdateSlide"
          loading={isLoading}
        >
          Update Slide
        </Button>
      </div>
    </div>
  );
};

export default ManageUpdateSlide;
