"use client";

import { Button } from "@/components/Buttons";
import { RootState } from "@/redux/store";
import { copyString, generateRealTimeId } from "@/utils/utilities";
import React, { useEffect, useState } from "react";
import { getJitsiJwt } from "@/services/jitsiToken.service";
import toast from "react-hot-toast";
import { FaSquareWhatsapp } from "react-icons/fa6";
import { MdEmail } from "react-icons/md";
import { useSelector } from "react-redux";

const ManageRealTimeView = () => {
  const [meetingId, setMeetingId] = useState<string>("");
  const { activeUser } = useSelector((state: RootState) => state.user);
  const [isLoading, setIsLoading] = useState<boolean>(false);

  useEffect(() => {
    const meeting = generateRealTimeId();
    setMeetingId(meeting);
  }, []);


  const onStart = async () => {
    setIsLoading(true);
    try {
      if (!activeUser) throw new Error("User not found");
      const jwt = await getJitsiJwt({
        firstName: activeUser.firstName,
        lastName: activeUser.lastName,
        email: activeUser.email,
        meetingId,
        userId: activeUser.id,
        avatar: activeUser.profileImage || "",
      });
      const meetingUrl = `https://meet.xolanibridge.com/${meetingId}?jwt=${jwt}`;
      window.open(meetingUrl, "_blank");
    } catch (err) {
      toast("Failed to start meeting");
    } finally {
      setIsLoading(false);
    }
  };

  const onCopy = async ({ value }: { value: string }) => {
    await copyString(value);
    toast("Copied to clipboard!");
  };

  // Detect mobile devices for WhatsApp sharing
  const isMobile = () => /iPhone|iPad|iPod|Android/i.test(navigator.userAgent);


  const shareOnWhatsApp = () => {
    const meetingUrl = `https://meet.xolanibridge.com/${meetingId}`;
    const message = encodeURIComponent(`Join my call: ${meetingUrl}`);
    window.open(
      `https://${isMobile() ? "api" : "web"}.whatsapp.com/send?text=${message}`,
    );
  };


  const shareViaEmail = () => {
    const meetingUrl = `https://meet.xolanibridge.com/${meetingId}`;
    const subject = encodeURIComponent("Join my call Session");
    const body = encodeURIComponent(
      `Click this link to join the call:\n\n ${meetingUrl}`,
    );
    window.location.href = `mailto:?subject=${subject}&body=${body}`;
  };

  return (
    <div className="relative w-[80vw] overflow-hidden rounded-lg bg-white px-8 py-10 sm:w-[400px]">
      <div>
        <p className="py-4 text-center text-xl font-medium text-grey">
          Call Link Created
        </p>
        <div className="mb-4 flex h-[45px] items-center rounded-[4px] border border-greyLine px-1">
          <p className="px-2 text-sm text-greySecondary">{`meet.xolanibridge.com/${meetingId}`}</p>
          <Button
            onClick={() => onCopy({ value: `https://meet.xolanibridge.com/${meetingId}` })}
            variant="lightGreen"
            className="ml-auto h-[35px] px-2"
          >
            Copy
          </Button>
        </div>
        <div>
          <p className="text-sm text-grey">Share to</p>
          <div className="flex items-center gap-2">
            <FaSquareWhatsapp
              size={50}
              onClick={shareOnWhatsApp}
              className="text-whatsapp cursor-pointer transition-all duration-200 ease-in-out hover:scale-105"
            />

            <MdEmail
              size={60}
              onClick={shareViaEmail}
              className="cursor-pointer text-blue-500 transition-all duration-200 ease-in-out hover:scale-105"
            />
          </div>
        </div>
      </div>
      <div className="mx-auto mt-8">
        <div className="mb-4 text-center text-sm text-red-500 font-medium">
          You must start a collaboration before guests can join the call.
        </div>
        <Button
          loading={isLoading}
          variant="primary"
          className="w-full"
          onClick={() => onStart()}
        >
          Start Collaboration
        </Button>
      </div>
    </div>
  );
};

export default ManageRealTimeView;
