"use client";

import { Button } from "@/components/Buttons";
import { SelectInput, TextInput } from "@/components/Inputs";
import { togglePopup } from "@/redux/global/globalSlice";
import { AppDispatch, RootState } from "@/redux/store";
import institutionService from "@/services/institution.service";
import { MANAGE_PATHOLOGIST_SCHEMA } from "@/utils/validationSchema/institution.schema";
import { Form, Formik } from "formik";
import React, { useState } from "react";
import toast from "react-hot-toast";
import { useDispatch, useSelector } from "react-redux";
import { inHousePathologistType } from "../constants";
import usePathologists from "../pathologists/hooks/usePathologists";
import { User } from "@/interfaces";

interface FormValues {
  email: string;
  firstName: string;
  lastName: string;
  type: string;
}

type Pathologist = {
  id: number;
  invitedAt: Date | string;
  status: string;
  type: string;
  pathologist: {
    id: number;
    verified: boolean;
    createdAt: Date | string;
    user: User;
    type: string;
  };
};

const ManagePathologists = ({
  pathologist,
  isType,
}: {
  pathologist?: Pathologist;
  isType?: boolean;
}) => {
  const reduxDispatch = useDispatch<AppDispatch>();
  const [isLoading, setIsLoading] = useState<boolean>();

  const { getPathologists, search } = usePathologists();

  const { activeInstitution } = useSelector((state: RootState) => state.user);

  const initialValues: FormValues = {
    email: isType ? pathologist?.pathologist?.user?.email || "" : "",
    firstName: isType ? pathologist?.pathologist?.user?.firstName || "" : "",
    lastName: isType ? pathologist?.pathologist?.user?.lastName || "" : "",
    type: isType ? pathologist?.type || "" : "",
  };
  const onHandleSubmit = async (values: FormValues) => {
    if (activeInstitution) {
      setIsLoading(true);
      institutionService
        .addPathologist({
          institutionId: activeInstitution.id,
          ...values,
        })
        .then((res) => {
          toast.success(res.message);
          setTimeout(() => {
            getPathologists(search);
          }, 1000);
          reduxDispatch(togglePopup({ key: "firstPopup", isOpen: false }));
        })
        .catch((error) => {
          console.log(error);
          toast.error(error?.message || error?.errors);
        })
        .finally(() => {
          setIsLoading(false);
        });
    }
  };
  const onHandleUpdateType = async (values: FormValues) => {
    if (activeInstitution && pathologist) {
      setIsLoading(true);
      institutionService
        .changePathologistType({
          institutionId: activeInstitution.id,
          pathologistId: pathologist?.id,
          type: values.type,
        })
        .then((res) => {
          toast.success(res.message);
          getPathologists(search);
          reduxDispatch(togglePopup({ key: "firstPopup", isOpen: false }));
        })
        .catch((error) => {
          console.log(error);
          toast.error(error?.message || error?.errors);
        })
        .finally(() => {
          setIsLoading(false);
        });
    }
  };

  return (
    <div className="flex h-[500px] w-[85vw] flex-col rounded-lg bg-white sm:w-[500px]">
      <div className="mb-3 px-3 pb-3 pt-8 text-center shadow-sm sm:px-8">
        <span className="text-xl font-semibold text-grey">
          Add InHouse Pathologists
        </span>
      </div>
      <div className="relative flex flex-1 flex-col overflow-auto px-3 pb-4 pt-2 sm:px-8">
        <p className="text text-center font-medium text-greySecondary">
          Get quick interpretations by adding you In-house Pathologists
        </p>

        <Formik
          initialValues={initialValues}
          validationSchema={MANAGE_PATHOLOGIST_SCHEMA}
          onSubmit={isType ? onHandleUpdateType : onHandleSubmit}
        >
          {() => (
            <Form className="mt-4 space-y-1" id="managePathologistForm">
              <TextInput
                name="firstName"
                type="text"
                label="First Name"
                disabled={isLoading || isType}
              />
              <TextInput
                name="lastName"
                type="text"
                label="Last Name"
                disabled={isLoading || isType}
              />
              <TextInput
                name="email"
                type="email"
                label="Email Address"
                disabled={isLoading || isType}
              />
              <SelectInput
                name="type"
                label="Type"
                options={inHousePathologistType}
                type="primary"
                placeholder=""
                className={{
                  content: "w-full",
                  trigger:
                    "h-[40px] w-full rounded-[4px] border-greyLine text-blackMain outline-none",
                }}
              />{" "}
            </Form>
          )}
        </Formik>
      </div>
      <div className="flex w-full justify-between border-t-2 border-greyStroke px-3 py-6 sm:px-8">
        <Button
          variant="gray"
          type="button"
          onClick={() =>
            reduxDispatch(togglePopup({ key: "firstPopup", isOpen: false }))
          }
        >
          Cancel
        </Button>
        <Button
          variant="primary"
          type="submit"
          form="managePathologistForm"
          loading={isLoading}
        >
          {isType ? "Update" : "Add"}
        </Button>
      </div>
    </div>
  );
};

export default ManagePathologists;
