import useReportStorage from "@/app/institution/view-studies/[studyId]/(components)/Report/hooks/useReportStorage";
import useSpeechRecognition from "@/app/institution/view-studies/[studyId]/(components)/Report/hooks/useSpeechRecognition";
import { Button } from "@/components/Buttons";
import { TextInput } from "@/components/Inputs";
import Loading from "@/components/Loading/Loading";
import { setExternalViewerData } from "@/redux/data/dataSlice";
import { setDrawer } from "@/redux/global/globalSlice";
import { AppDispatch, RootState } from "@/redux/store";
import slideService from "@/services/slide.service";
import { EXTERNAL_VIEWER_REPORT_SCHEMA } from "@/utils/validationSchema/auth.schema";
import { Form, Formik } from "formik";
import React, { useState } from "react";
import toast from "react-hot-toast";
import { MdMic, MdMicOff, MdOutlineCalendarToday } from "react-icons/md";
import { useDispatch, useSelector } from "react-redux";

interface FormValues {
  opinion: string;
}
const ExternalReport = () => {
  const { externalViewerData } = useSelector((state: RootState) => state.data);
  const [loading, setLoading] = useState<boolean>(false);

  const reduxDispatch = useDispatch<AppDispatch>();

  const { storedReport, updateStoredReport, clearReport, reportLoading } =
    useReportStorage(externalViewerData?.reportId as string);

  const { isListening, activeField, runSpeechRecog } =
    useSpeechRecognition(updateStoredReport);

  const initialValues: FormValues = {
    opinion: storedReport?.opinion || "",
  };

  const { report, slideImage } = externalViewerData;

  const onSubmit = async (values: FormValues) => {
    setLoading(true);

    slideService
      .externalOpinion({
        shareId: externalViewerData?.shareId,
        body: {
          email: externalViewerData?.accessEmail,
          code: externalViewerData?.accessCode,
          text: values.opinion,
          reportId: externalViewerData?.reportId,
        },
      })
      .then((res) => {
        toast.success(res?.message);
        clearReport();
        reduxDispatch(setDrawer({ isOpen: false }));
        reduxDispatch(setExternalViewerData({}));
      })
      .catch((error) => {
        console.log(error);
        toast.error(error?.message || error?.errors);
        setLoading(false);
      })
      .finally(() => setLoading(false));
  };

  return (
    <div className="h-full overflow-y-auto px-2 py-5">
      <div className="mb-8">
        <p className="text-center text-[20px] font-semibold text-grey">
          Second Opinion
        </p>
      </div>
      <div className="relative space-y-4">
        {reportLoading ? (
          <Loading
            className={{
              loader: "!w-[40px]",
              loaderText: "text-lg text-grey",
            }}
            loaderText="Loading report..."
          />
        ) : (
          <React.Fragment>
            {loading && (
              <Loading
                className={{
                  loader: "!w-[40px]",
                  loaderText: "text-lg text-grey",
                }}
                loaderText="Loading ..."
              />
            )}
            <Formik
              initialValues={initialValues}
              validationSchema={EXTERNAL_VIEWER_REPORT_SCHEMA}
              onSubmit={onSubmit}
            >
              {({ setFieldValue, values }) => {
                return (
                  <Form className="space-y-4">
                    <div className="relative">
                      <p className="mb-1 text-sm font-medium text-grey">
                        Your Opinion
                      </p>
                      <TextInput
                        name="opinion"
                        type="textarea"
                        rows={6}
                        onChange={(e) => {
                          updateStoredReport("opinion", e.target.value);
                        }}
                      />
                      <Button
                        title={
                          isListening && activeField === "opinion"
                            ? "Stop"
                            : "Tap and Talk"
                        }
                        variant="secondary"
                        type="button"
                        className={`${isListening && activeField === "opinion" ? "animate-pulse border-red-500 text-red-500" : ""} absolute bottom-1 right-1 flex size-[38px] items-center justify-center rounded-full p-0`}
                        onClick={() => {
                          runSpeechRecog(
                            "opinion",
                            setFieldValue,
                            values.opinion,
                          );
                        }}
                      >
                        {isListening && activeField === "opinion" ? (
                          <MdMicOff size={20} />
                        ) : (
                          <MdMic size={20} />
                        )}
                      </Button>
                    </div>

                    <div className="!mt-14 flex justify-end gap-2 max-md:flex-wrap">
                      <Button
                        variant="secondary"
                        type="submit"
                        onClick={() => clearReport(setFieldValue)}
                      >
                        Send for Opinion
                      </Button>
                    </div>
                  </Form>
                );
              }}
            </Formik>
          </React.Fragment>
        )}
      </div>
      <div className="my-8 border-t-2 pt-6">
        <p className="text-center text-[18px] font-semibold text-grey">
          Slide Details
        </p>
      </div>
      <div className="space-y-2">
        <div className="grid grid-cols-12 gap-2 md:gap-6">
          <div className="col-span-6">
            <p className="mb-1 text-sm font-medium text-grey">Date of Birth</p>
            <p className="m-0 flex h-[45px] items-center justify-between truncate rounded-[4px] border border-greyLine px-4 text-greySecondary">
              {slideImage?.dob || "-- N/A --"}
              <MdOutlineCalendarToday
                size={20}
                className="font-light text-greyLine"
              />
            </p>
          </div>
          <div className="col-span-6">
            <p className="mb-1 text-sm font-medium text-grey">
              Patient's Gender
            </p>
            <div className="grid grid-cols-12 gap-2 md:gap-3">
              <span
                className={`col-span-4 flex h-[45px] items-center justify-center border px-6 text-greySecondary ${slideImage?.patientGender === "M" ? "border-primaryMain" : "border-greyLine"} rounded-[4px]`}
              >
                M
              </span>
              <span
                className={`col-span-4 flex h-[45px] items-center justify-center border px-6 text-greySecondary ${slideImage?.patientGender === "F" ? "border-primaryMain" : "border-greyLine"} rounded-[4px]`}
              >
                F
              </span>
              <span
                className={`col-span-4 flex h-[45px] items-center justify-center border px-6 text-greySecondary ${
                  slideImage?.patientGender !== "M" &&
                  slideImage?.patientGender !== "F"
                    ? "border-primaryMain"
                    : "border-greyLine"
                } rounded-[4px]`}
              >
                O
              </span>
            </div>
          </div>
        </div>
        <div className="">
          <p className="mb-1 text-sm font-medium text-grey">Clinical History</p>
          <div className="boder-greyLine max-h-[400px] min-h-[200px] overflow-y-scroll rounded-[4px] border p-4 text-greySecondary">
            <p className="m-0">{slideImage?.clinicalHistory || "-- N/A --"}</p>
          </div>
        </div>

        <div className="">
          <p className="mb-1 text-sm font-medium text-grey">
            Macroscopic Description
          </p>
          <div className="boder-greyLine max-h-[400px] min-h-[200px] overflow-y-scroll rounded-[4px] border p-4 text-greySecondary">
            <p className="m-0">
              {slideImage?.macroscopicDescription ||
                slideImage?.comment ||
                "-- N/A --"}
            </p>
          </div>
        </div>
        <div className="">
          <p className="mb-1 text-sm font-medium text-grey">
            Nature of Specimen
          </p>
          <div className="boder-greyLine max-h-[200px] min-h-[100px] overflow-y-scroll rounded-[4px] border p-4 text-greySecondary">
            <p className="m-0">{slideImage?.nature || "-- N/A --"}</p>
          </div>
        </div>
        <div className="">
          <p className="mb-1 text-sm font-medium text-grey">Microscopy</p>
          <div className="boder-greyLine max-h-[400px] min-h-[200px] overflow-y-scroll rounded-[4px] border p-4 text-greySecondary">
            <p className="m-0">{report?.microscopy || "-- N/A --"}</p>
          </div>
        </div>
        <div className="">
          <p className="mb-1 text-sm font-medium text-grey">Final Diagnosis</p>
          <div className="boder-greyLine max-h-[400px] min-h-[200px] overflow-y-scroll rounded-[4px] border p-4 text-greySecondary">
            <p className="m-0">{report?.diagnosis || "-- N/A --"}</p>
          </div>
        </div>
        <div className="">
          <p className="mb-1 text-sm font-medium text-grey">
            Comment (Recommendation)
          </p>
          <div className="boder-greyLine max-h-[400px] min-h-[200px] overflow-y-scroll rounded-[4px] border p-4 text-greySecondary">
            <p className="m-0">{report?.recommendations || "-- N/A --"}</p>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ExternalReport;
