"use client";

import { Button } from "@/components/Buttons";
import authService from "@/services/auth.service";
import { getAuthEmail } from "@/utils/sessions/session.utils";
import { VERIFY_SCHEMA } from "@/utils/validationSchema/auth.schema";
import { useFormik } from "formik";
import { useRouter } from "next/navigation";
import React, { useEffect, useState } from "react";
import toast from "react-hot-toast";
import OTPInput from "react-otp-input";

interface FormValues {
  email: string;
  otp: string;
}

const initialValues: FormValues = {
  otp: "",
  email: "",
};
const Page = () => {
  const router = useRouter();
  const [authEmail, setAuthEmail] = useState<string | null>("");
  const [isLoading, setIsLoading] = useState<boolean>(false);
  const [isVerificationSuccessful, setIsVerificationSuccessful] =
    useState<boolean>(false);
  const [isSigningIn, setIsSigningIn] = useState<boolean>(false);

  const onSubmit = (values: FormValues) => {
    setIsLoading(true);
    authService
      .verifySignUp(values)
      .then(() => {
        setIsVerificationSuccessful(true);
        toast.success("Your email has been verified successfully.");
      })
      .catch((error) => {
        toast.error(error?.message || error?.errors);
      })
      .finally(() => setIsLoading(false));
  };

  const onResend = () => {
    if (authEmail) {
      setIsLoading(true);

      authService
        .resendVerifySignup({ email: authEmail })
        .then(() => {
          toast.success("An OTP has been sent to your email.");
        })
        .catch((error) => {
          toast.error(error?.message || error?.errors);
        })
        .finally(() => setIsLoading(false));
    }
  };

  const onSignIn = () => {
    setIsSigningIn(true);
    router.push("/auth/login");

    setTimeout(() => {
      setIsSigningIn(false);
    }, 2000);
  };

  const formik = useFormik({
    initialValues,
    validationSchema: VERIFY_SCHEMA,
    onSubmit: onSubmit,
  });

  const { handleSubmit, errors, touched, setFieldValue, values } = formik;

  useEffect(() => {
    const storedEmail = getAuthEmail();
    setAuthEmail(storedEmail);

    if (!storedEmail) {
      router.push("/auth/login");
    } else {
      setFieldValue("email", storedEmail);
    }
  }, [router]);

  if (isVerificationSuccessful) {
    return (
      <div className="mx-auto w-[90vw] rounded-xl bg-white sm:w-[560px]">
        <div className="px-4 py-10 text-center sm:px-10">
          <h2 className="m-0 text-[20px] font-medium text-grey">
            Verification Successful
          </h2>
          <div className="my-10 flex justify-center">
            <Button loading={isSigningIn} variant="primary" onClick={onSignIn}>
              {!isSigningIn ? "Sign In to Continue" : "Signing you in..."}
            </Button>
          </div>
        </div>
      </div>
    );
  }
  return (
    <>
      <div className="mx-auto w-[85vw] rounded-xl bg-white px-8 sm:w-[460px]">
        <div>
          <div className="px-4 py-4 text-center">
            <p className="m-0 text-[28px] font-medium text-blackMain sm:text-[32px]">
              Verify OTP
            </p>
            <p className="mb-0 text-[14px] font-normal text-grey">
              {`Just to confirm it's really you, please check your email, we have
            sent a verification code.`}
            </p>
          </div>
          <div className="px-4 pb-6 pt-2 sm:px-6">
            <p className="mb-4 text-center text-[14px] font-normal text-grey">
              Please enter the verification code sent to
              <br />
              {authEmail || " your email"}
            </p>
            <form onSubmit={handleSubmit}>
              <div className="relative flex items-center"></div>
              <OTPInput
                value={values.otp}
                inputStyle="!w-[55px] h-[55px]  text-lg border-2 border-greyLine outline-primary  rounded-md text-blackMain font-medium"
                containerStyle="flex justify-center gap-4 mt-4 px-4"
                numInputs={6}
                onChange={(otp: string) => setFieldValue("otp", otp)}
                renderInput={(props) => <input {...props} />}
              />
              {errors.otp && touched.otp ? (
                <p className="mb-0 text-center text-[12px] font-medium text-[#ff0000]">
                  {errors.otp}
                </p>
              ) : null}

              <div className="mt-6">
                <Button
                  variant="primary"
                  type="submit"
                  className="w-full"
                  loading={isLoading}
                >
                  Verify
                </Button>
              </div>
            </form>
            <div className="mt-4 pb-6">
              <p className="mb-4 text-center text-sm text-grey">
                {`Didn't receive a verification code?`} <br />
                <span
                  onClick={() => onResend()}
                  className="cursor-pointer font-medium text-primary"
                >
                  Resend Code
                </span>
              </p>
            </div>
          </div>
        </div>
      </div>
    </>
  );
};

export default Page;
