"use client";

import { Button } from "@/components/Buttons";
import { TextInput } from "@/components/Inputs";
import authService from "@/services/auth.service";
import { getAuthEmail } from "@/utils/sessions/session.utils";
import { RESET_SCHEMA } from "@/utils/validationSchema/auth.schema";
import { Form, Formik } from "formik";
import Link from "next/link";
import { useRouter } from "next/navigation";
import React, { useEffect, useState } from "react";
import toast from "react-hot-toast";

interface FormValues {
  email: string;
  password: string;
  confirmPassword: string;
}

const initialValues: FormValues = {
  email: "",
  password: "",
  confirmPassword: "",
};

const Page = () => {
  const router = useRouter();
  const [isLoading, setIsLoading] = useState<boolean>(false);

  useEffect(() => {
    const storedEmail = getAuthEmail();

    if (!storedEmail) {
      toast.error("No reset email found");
      router.push("/auth/login");
    }
  }, []);

  const onSubmit = (values: FormValues) => {
    setIsLoading(true);
    const storedEmail = getAuthEmail();
    if (!storedEmail) {
      toast.error("No reset email found");
      router.push("/auth/login");
    } else {
      authService
        .resetPassword({
          ...values,
          email: storedEmail,
        })
        .then(() => {
          toast.success("Your password have been updated successfully. 🎉");
          router.push("/auth/login");
        })
        .catch((error) => {
          toast.error(error?.message || error?.errors);
        })
        .finally(() => setIsLoading(false));
    }
  };

  return (
    <>
      <div className="mx-auto w-[85vw] rounded-xl bg-white sm:w-[560px]">
        <div>
          <div className="px-4 py-4 text-center">
            <p className="m-0 text-[28px] font-medium text-blackMain sm:text-[32px]">
              Reset Password
            </p>
            <p className="mb-0 text-[14px] font-normal text-grey">
              Set your new password.{" "}
            </p>
          </div>
          <div className="px-4 pb-16 pt-6 sm:px-12">
            <Formik
              initialValues={initialValues}
              validationSchema={RESET_SCHEMA}
              onSubmit={onSubmit}
            >
              {() => (
                <Form className="flex flex-col gap-3">
                  <TextInput
                    name="password"
                    type="password"
                    label="New Password"
                    placeholder="********"
                    disabled={isLoading}
                  />
                  <TextInput
                    name="confirmPassword"
                    type="password"
                    label="Confirm New Password"
                    placeholder="********"
                    disabled={isLoading}
                  />
                  <p className="text-xs text-greyLight">
                    Password Hint: A capital letter[A-Z], number[123...],
                    special character[@!%$*&] and at least six(6) digits.{" "}
                  </p>

                  <div className="mt-10 flex justify-end">
                    <Button variant="primary" type="submit" loading={isLoading}>
                      Reset Password
                    </Button>
                  </div>
                </Form>
              )}
            </Formik>
          </div>
        </div>
      </div>
      <div className="p-6 text-center">
        <p className="mb-0 text-sm font-medium text-grey">
          Return to &nbsp;
          <Link href="/auth/login" className="text-primary hover:text-primary">
            Sign In
          </Link>
        </p>
      </div>
    </>
  );
};

export default Page;
