"use client";

import React, { useState } from "react";
import { PhoneInput, SelectInput, TextInput } from "@/components/Inputs";

import { Form, Formik } from "formik";

import { useRouter } from "next/navigation";
import {
  REGISTER_ONE_SCHEMA,
  REGISTER_TWO_SCHEMA,
} from "@/utils/validationSchema/auth.schema";
import { Button } from "@/components/Buttons";
import Link from "next/link";
import { allCountries } from "country-region-data";
import { institutionTypes } from "../contants";
import { setAuthEmail } from "@/utils/sessions/session.utils";
import authService from "@/services/auth.service";
import toast from "react-hot-toast";
import { parsePhoneNumber } from "react-phone-number-input";

interface FormValues {
  institutionName: string;
  registrationNumber: string;
  institutionType?:
    | "MEDICAL_CENTER"
    | "DIAGNOSTIC_CENTER"
    | "HOSPITAL"
    | "GROUP_OF_RADIOLOGISTS"
    | "OTHERS";
  institutionPhone: string;
  institutionEmail: string;
  institutionCountry: string;
  institutionAddress: string;
  firstName: string;
  lastName: string;
  phone: string;
  email: string;
  password: string;
  cPassword: string;
  additionalField1?: string; // Add fields for stages 3 and 4
  additionalField2?: string;
}

const Page = () => {
  const [stage, setStage] = useState<number>(1);
  const [isLoading, setIsLoading] = useState<boolean>(false);
  // const [isTermsRead, setIsTermsRead] = useState<boolean>(false);
  const router = useRouter();

  const initialValues: FormValues = {
    institutionName: "",
    registrationNumber: "",
    institutionType: undefined,
    institutionPhone: "",
    institutionEmail: "",
    institutionCountry: "",
    institutionAddress: "",
    firstName: "",
    lastName: "",
    phone: "",
    email: "",
    password: "",
    cPassword: "",
  };

  const [formValues, setFormValues] = useState<FormValues>(initialValues);

  const handleStageSubmit = (
    values: Partial<FormValues>,
    nextStage: number,
  ) => {
    setFormValues((prevValues) => ({
      ...prevValues,
      ...values,
    }));
    setStage(nextStage);
  };

  const countryData = allCountries
    ?.map((country) => ({
      value: country[1],
      label: country[0],
    }))
    .filter((count) => count.value === "NG");

  const handleSubmit = (
    values: FormValues,
    { resetForm }: { resetForm: () => void },
  ) => {
    setIsLoading(true);
    setAuthEmail(values.email);
    const instPhone = values.institutionPhone
      ? parsePhoneNumber(values.institutionPhone)
      : null;
    const phone = values.phone ? parsePhoneNumber(values.phone) : null;

    if (!instPhone || !phone) {
      toast.error("Invalid phone number. Please enter a valid phone number.");
      setIsLoading(false);
      return;
    }
    authService
      .createInstitution({
        ...values,
        institutionPhone: {
          phoneCode: `+${instPhone.countryCallingCode}`,
          phoneNumber: instPhone.nationalNumber,
        },
        phone: {
          phoneCode: `+${phone.countryCallingCode}`,
          phoneNumber: phone.nationalNumber,
        },
      })
      .then(() => {
        toast("An OTP has been sent to complete your signup process.");
        resetForm();
        setFormValues(initialValues);
        router.push("/auth/verify");
      })
      .catch((error) => {
        toast.error(
          error?.message ||
            error?.errors ||
            "Oops! Something went wrong, please try again.",
        );
      })
      .finally(() => {
        setIsLoading(false);
      });
  };

  return (
    <>
      <div className="mx-auto w-[85vw] rounded-xl bg-white p-4 sm:w-[560px] sm:px-12">
        <div>
          <div className="pb-5 pt-4 text-center sm:pt-8">
            <p className="mb-2 text-[16px] font-medium text-blackMain">
              Register as an Institution (Hospital, Lab, Pharmacy,
              Clinic...){" "}
            </p>
            <p className="mx-auto mb-0 max-w-[400px] text-[12px] font-normal text-grey">
              {stage === 1 &&
                "Let’s get you started with your institution’s registration."}
              {stage === 2 &&
                "Provide the details of the main contact person for your institution"}
            </p>
            <div className="mt-4 flex justify-center gap-2">
              <div
                className={`${stage === 1 ? "bg-primaryMain" : "bg-lightGreenLight"} h-[7px] w-[100px] rounded-xl`}
              ></div>
              <div
                className={`${stage === 2 ? "bg-primaryMain" : "bg-lightGreenLight"} h-[7px] w-[100px] rounded-xl`}
              ></div>
            </div>
          </div>
          {stage === 1 && (
            <div className="pb-10">
              <p className="mb-1 text-[16px] font-medium text-grey">
                Institution Information
              </p>
              <Formik
                initialValues={formValues}
                validationSchema={REGISTER_ONE_SCHEMA}
                onSubmit={(values) => handleStageSubmit(values, 2)}
              >
                {() => (
                  <Form className="flex flex-col gap-3">
                    <TextInput
                      name="institutionName"
                      type="text"
                      label="Institution Name"
                      placeholder="John"
                      disabled={isLoading}
                    />
                    <TextInput
                      name="registrationNumber"
                      type="text"
                      label="Registration Number (Optional)"
                      placeholder="ABC123"
                      disabled={isLoading}
                    />
                    <SelectInput
                      name="institutionType"
                      label="Type of Institution"
                      options={institutionTypes}
                      disabled={isLoading}
                      type="primary"
                      placeholder=""
                      className={{
                        content: "w-full",
                        trigger:
                          "h-[40px] w-full rounded-[4px] border-greyLine text-blackMain outline-none",
                      }}
                    />

                    <PhoneInput name="institutionPhone" label="Phone No." />

                    <TextInput
                      name="institutionEmail"
                      type="email"
                      label="Institution Email"
                      placeholder="Enter your email"
                      disabled={isLoading}
                    />
                    <SelectInput
                      name="institutionCountry"
                      label="Institution Country"
                      options={countryData}
                      disabled={isLoading}
                      type="primary"
                      placeholder=""
                      className={{
                        content: "w-full",
                        trigger:
                          "h-[40px] w-full rounded-[4px] border-greyLine text-blackMain outline-none",
                      }}
                    />
                    <TextInput
                      name="institutionAddress"
                      type="text"
                      label="Address"
                      placeholder="XYZ Street..."
                      disabled={isLoading}
                    />

                    <div className="mt-10 flex justify-end">
                      <Button
                        type="submit"
                        variant="primary"
                        loading={isLoading}
                      >
                        Next
                      </Button>
                    </div>
                  </Form>
                )}
              </Formik>
            </div>
          )}
          {stage === 2 && (
            <div className="pb-10">
              <p className="mb-1 text-[16px] font-medium text-grey">
                {`Who will be managing your institution's account`}
              </p>
              <Formik
                initialValues={formValues}
                validationSchema={REGISTER_TWO_SCHEMA}
                onSubmit={handleSubmit}
              >
                {() => (
                  <Form className="flex flex-col gap-3">
                    <TextInput
                      name="firstName"
                      type="text"
                      label="First Name"
                      placeholder="John"
                      disabled={isLoading}
                    />
                    <TextInput
                      name="lastName"
                      type="text"
                      label="Last Name"
                      placeholder="Doe"
                      disabled={isLoading}
                    />
                    <TextInput
                      name="email"
                      type="email"
                      label="Work Email"
                      placeholder="john.doe@gmail.com"
                      disabled={isLoading}
                    />

                    <PhoneInput name="phone" label="Phone Number" />

                    <div>
                      <TextInput
                        name="password"
                        type="password"
                        label="Create Password"
                        placeholder="********"
                        disabled={isLoading}
                      />
                      <span className="text-xs text-greyLight">
                        Password Hint: At least six (6) digits.
                      </span>
                    </div>

                    <TextInput
                      name="cPassword"
                      type="password"
                      label="Confirm Password"
                      placeholder="********"
                      disabled={isLoading}
                    />

                    <div className="mt-10 flex justify-between">
                      <Button variant="gray" onClick={() => setStage(1)}>
                        Previous
                      </Button>
                      <Button
                        variant="primary"
                        type="submit"
                        loading={isLoading}
                      >
                        Next
                      </Button>
                    </div>
                  </Form>
                )}
              </Formik>
            </div>
          )}
        </div>
      </div>
      <div className="p-6 text-center">
        <p className="mb-0 text-sm font-medium text-grey">
          Already have an account? &nbsp;
          <Link href="/auth/login" className="text-primary hover:text-primary">
            Sign In here
          </Link>
        </p>
      </div>
    </>
  );
};

export default Page;
