"use client";

import React, { useState } from "react";
import { PhoneInput, SelectInput, TextInput } from "@/components/Inputs";
import { Form, Formik } from "formik";
import { MdOutlineArrowBack } from "react-icons/md";
import { useRouter } from "next/navigation";
import { Button } from "@/components/Buttons";
import Link from "next/link";
import { allCountries } from "country-region-data";
import {
  REGISTER_PATHOLOGIST_ONE_SCHEMA,
  REGISTER_PATHOLOGIST_TWO_SCHEMA,
} from "@/utils/validationSchema/auth.schema";
import { setAuthEmail } from "@/utils/sessions/session.utils";
import { parsePhoneNumber } from "react-phone-number-input";
import toast from "react-hot-toast";
import authService from "@/services/auth.service";

interface FormValues {
  firstName: string;
  lastName: string;
  phone: string;
  email: string;
  country: string;
  password: string;
  cPassword: string;
  address: string;
  additionalField1?: string; // Add fields for stages 3 and 4
  additionalField2?: string;
}

const Page = () => {
  const [stage, setStage] = useState<number>(1);
  const [isLoading, setIsLoading] = useState<boolean>(false);
  // const [isTermsRead, setIsTermsRead] = useState<boolean>(false);
  const router = useRouter();

  const initialValues: FormValues = {
    firstName: "",
    lastName: "",
    phone: "",
    email: "",
    password: "",
    cPassword: "",
    address: "",
    country: "",
  };

  const [formValues, setFormValues] = useState<FormValues>(initialValues);

  const handleStageSubmit = (
    values: Partial<FormValues>,
    nextStage: number,
  ) => {
    setFormValues((prevValues) => ({
      ...prevValues,
      ...values,
    }));
    setStage(nextStage);
  };

  const countryData = allCountries
    ?.map((country) => ({
      value: country[1],
      label: country[0],
    }))
    .filter((count) => count.value === "NG");

  const handleSubmit = (
    values: FormValues,
    { resetForm }: { resetForm: () => void },
  ) => {
    setIsLoading(true);
    setAuthEmail(values.email);

    const phone = values.phone ? parsePhoneNumber(values.phone) : null;

    if (!phone) {
      toast.error("Invalid phone number. Please enter a valid phone number.");
      setIsLoading(false);
      return;
    }
    authService
      .createPathologist({
        ...values,

        phone: {
          phoneCode: `+${phone.countryCallingCode}`,
          phoneNumber: phone.nationalNumber,
        },
      })
      .then(() => {
        toast("An OTP has been sent to complete your signup process.");
        resetForm();
        setFormValues(initialValues);
        router.push("/auth/verify");
      })
      .catch((error) => {
        toast.error(
          error?.message ||
            error?.errors ||
            "Oops! Something went wrong, please try again.",
        );
      })
      .finally(() => {
        setIsLoading(false);
      });
  };

  return (
    <>
      <div className="mx-auto w-[90vw] rounded-xl bg-white p-4 sm:w-[560px] sm:px-12">
        {[1, 2].includes(stage) && (
          <div>
            <div className="pb-5 pt-8 text-center">
              <p className="mb-2 text-[16px] font-medium text-blackMain">
                Register as a Specialist/Pathologist{" "}
              </p>
              <p className="mx-auto mb-0 max-w-[400px] text-[12px] font-normal text-grey">
                {stage === 1 && (
                  <>
                    Let’s get you started with your i registration. To register
                    with a different account,{" "}
                    <Link
                      href={"/auth/register"}
                      className="text-primary hover:text-primary"
                    >
                      {" "}
                      Click here
                    </Link>
                  </>
                )}
                {stage === 2 && "Provide security details"}
              </p>
              <div className="mt-4 flex justify-center gap-2">
                <div
                  className={`${stage === 1 ? "bg-primaryMain" : "bg-lightGreenLight"} h-[7px] w-[50px] rounded-xl`}
                ></div>
                <div
                  className={`${stage === 2 ? "bg-primaryMain" : "bg-lightGreenLight"} h-[7px] w-[50px] rounded-xl`}
                ></div>
                {/*
                <div
                  className={`${stage === 3 ? "bg-primaryMain" : "bg-lightGreenLight"} h-[7px] w-[50px] rounded-xl`}
                ></div> */}
              </div>
            </div>
            {stage === 1 && (
              <div className="pb-10">
                <p className="mb-1 text-[16px] font-medium text-grey">
                  Personal Information
                </p>
                <Formik
                  initialValues={formValues}
                  validationSchema={REGISTER_PATHOLOGIST_ONE_SCHEMA}
                  onSubmit={(values) => handleStageSubmit(values, 2)}
                >
                  {() => (
                    <Form className="flex flex-col gap-3">
                      <TextInput
                        name="firstName"
                        type="text"
                        label="First Name"
                        placeholder="John"
                        disabled={isLoading}
                      />
                      <TextInput
                        name="lastName"
                        type="text"
                        label="Last Name"
                        placeholder="Doe"
                        disabled={isLoading}
                      />
                      <TextInput
                        name="email"
                        type="email"
                        label="Email"
                        placeholder="Enter your email"
                        disabled={isLoading}
                      />

                      <SelectInput
                        name="country"
                        label="Country of Residence"
                        options={countryData}
                        disabled={isLoading}
                        type="primary"
                        placeholder=""
                        className={{
                          content: "w-full",
                          trigger:
                            "h-[40px] w-full rounded-[4px] border-greyLine text-blackMain outline-none",
                        }}
                      />

                      <PhoneInput name="phone" label="Phone No." />

                      {/* <SelectInput
                        name="gender"
                        label="Gender"
                        options={[
                          {
                            value: "MALE",
                            label: "Male",
                          },
                          {
                            value: "FEMALE",
                            label: "Female",
                          },
                        ]}
                        disabled={isLoading}
                        type="primary"
                        placeholder=""
                        className={{
                          content: "w-full",
                          trigger:
                            "h-[40px] w-full rounded-[4px] border-greyLine text-blackMain outline-none",
                        }}
                      /> */}

                      <TextInput
                        name="address"
                        type="text"
                        label="Address"
                        placeholder="XYZ Street..."
                        disabled={isLoading}
                      />

                      <div className="mt-10 flex justify-end">
                        <Button
                          variant="primary"
                          type="submit"
                          loading={isLoading}
                        >
                          Next
                        </Button>
                      </div>
                    </Form>
                  )}
                </Formik>
              </div>
            )}
            {stage === 2 && (
              <div>
                <div className="relative py-12">
                  <button
                    onClick={() => setStage(1)}
                    className="absolute left-[0px] top-[15px] flex items-center gap-1 text-[14px] font-medium text-primary"
                  >
                    <MdOutlineArrowBack size={18} /> Go back
                  </button>
                  <Formik
                    initialValues={formValues}
                    validationSchema={REGISTER_PATHOLOGIST_TWO_SCHEMA}
                    onSubmit={handleSubmit}
                  >
                    {() => (
                      <Form className="flex flex-col gap-3">
                        <TextInput
                          name="password"
                          type="password"
                          label="Password"
                          placeholder="********"
                          disabled={isLoading}
                        />

                        <TextInput
                          name="cPassword"
                          type="password"
                          label="Confirm Password"
                          placeholder="********"
                          disabled={isLoading}
                        />

                        <div className="mt-10">
                          <Button
                            variant="primary"
                            type="submit"
                            loading={isLoading}
                          >
                            Next
                          </Button>
                        </div>
                      </Form>
                    )}
                  </Formik>
                </div>
              </div>
            )}
            {/* {stage === 2 && (
              <div className="pb-10">
                <p className="mb-1 text-[16px] font-medium text-grey">
                  Personal Information
                </p>
                <Formik
                  initialValues={formValues}
                  validationSchema={REGISTER_PATHOLOGIST_TWO_SCHEMA}
                  onSubmit={(values) => handleStageSubmit(values, 1)}
                >
                  {() => (
                    <Form className="flex flex-col gap-3">
                      <TextInput
                        name="mlNumber"
                        type="text"
                        label="Medical License Number"
                        placeholder="ABC123"
                        disabled={isLoading}
                      />
                      <SelectInput
                        name="specialty"
                        label="Medical Specialty"
                        options={[
                          {
                            value: "MALE",
                            label: "Male",
                          },
                          {
                            value: "FEMALE",
                            label: "Female",
                          },
                        ]}
                        disabled={isLoading}
                        type="primary"
                        placeholder=""
                        className={{
                          content: "w-full",
                          trigger:
                            "h-[40px] w-full rounded-[4px] border-greyLine text-blackMain outline-none",
                        }}
                      />

                      <TextInput
                        name="experience"
                        type="text"
                        label="Years of Experience"
                        placeholder="search"
                        disabled={isLoading}
                      />

                      <TextInput
                        name="address"
                        type="text"
                        label="Work Addressr"
                        placeholder="XYZ Strret"
                        disabled={isLoading}
                      />

                      <TextInput
                        name="contactEmail"
                        type="email"
                        label="Contact Person Email Address"
                        placeholder="Enter your email"
                        disabled={isLoading}
                      />

                      <div className="mt-10 flex justify-between">
                        <Button
                          variant="gray"
                          onClick={() => setStage(1)}
                          disabled={isLoading}
                        >
                          Previous
                        </Button>
                        <Button
                          type="submit"
                          variant="primary"
                          loading={isLoading}
                        >
                          Next
                        </Button>
                      </div>
                    </Form>
                  )}
                </Formik>
              </div>
            )}
            {stage === 3 && (
              <div className="pb-10">
                <p className="mb-1 text-[16px] font-medium text-grey">
                  Operational Information
                </p>
                <Formik
                  initialValues={formValues}
                  // validationSchema={REGISTER_PATHOLOGIST_ONE_SCHEMA}
                  onSubmit={(values) => handleStageSubmit(values, 4)}
                >
                  {() => (
                    <Form className="flex flex-col gap-3">
                      <TextInput
                        name="services"
                        type="text"
                        label="Services Offered"
                        placeholder="Enter the services offered"
                        disabled={isLoading}
                      />
                      <TextInput
                        name="staff"
                        type="text"
                        label="Number of Staff"
                        placeholder="e.g 10, 27, 132"
                        disabled={isLoading}
                      />

                      <div className="mt-10 flex justify-between">
                        <Button
                          variant="gray"
                          onClick={() => setStage(2)}
                          disabled={isLoading}
                        >
                          Previous
                        </Button>
                        <Button
                          variant="primary"
                          type="submit"
                          loading={isLoading}
                        >
                          Next
                        </Button>
                      </div>
                    </Form>
                  )}
                </Formik>
              </div>
            )} */}
          </div>
        )}
      </div>
      <div className="p-6 text-center">
        <p className="mb-0 text-sm font-medium text-grey">
          Already have an account? &nbsp;
          <Link href="/auth/login" className="text-primary hover:text-primary">
            Sign In here
          </Link>
        </p>
      </div>
    </>
  );
};

export default Page;
