"use client";

import { Button } from "@/components/Buttons";
import { TextInput } from "@/components/Inputs";
import { setPopupChildren, togglePopup } from "@/redux/global/globalSlice";
import { AppDispatch } from "@/redux/store";
import { setActiveUser } from "@/redux/user/userSlice";
import authService from "@/services/auth.service";
import userService from "@/services/user.service";
import {
  setAuthEmail,
  setSessionUserData,
  setToken,
} from "@/utils/sessions/session.utils";
import { LOGIN_SCHEMA } from "@/utils/validationSchema/auth.schema";
import { Form, Formik } from "formik";
import Link from "next/link";
import { useRouter } from "next/navigation";
import React, { useState } from "react";
import toast from "react-hot-toast";
import { useDispatch } from "react-redux";

interface FormValues {
  email: string;
  password: string;
}

const initialValues: FormValues = {
  email: "",
  password: "",
};
const Page = () => {
  const router = useRouter();
  const reduxDispatch = useDispatch<AppDispatch>();
  const [isLoading, setIsLoading] = useState<boolean>(false);

  const getUserDetails = async () => {
    setIsLoading(true);
    userService
      .getUserDetails()
      .then((res) => {
        if (res?.data) {
          setSessionUserData(res.data);
          reduxDispatch(setActiveUser(res.data));
          router.push("/auth/account-select");
        }
        toast.success(res.message);
      })
      .catch((error) => {
        console.log(error);
        toast.error(error?.message || error?.errors);
        setIsLoading(false);
      })
      .finally(() => {
        setIsLoading(false);
      });
  };

  const handleLogin = async (values: FormValues) => {
    setIsLoading(true);
    authService
      .login(values)
      .then((res) => {
        setToken(res.data?.token as string);
        getUserDetails();
      })
      .catch((error) => {
        console.log(error);
        toast.error(error?.message || error?.errors);
        if (
          error?.message ===
          "User has not yet been verified. Pls proceed to verify your email"
        ) {
          setAuthEmail(values.email);
          router.push("/auth/verify");
        }
        setIsLoading(false);
      });
  };

  const onCreateAccount = () => {
    reduxDispatch(
      setPopupChildren({
        key: "firstPopup",
        componentKey: "createAccountTypes",
      }),
    );
    reduxDispatch(togglePopup({ key: "firstPopup", isOpen: true }));
  };

  return (
    <div className="mx-auto w-[85vw] rounded-xl bg-white sm:w-[560px]">
      <div>
        <div className="px-4 py-4 text-center sm:px-5">
          <p className="m-0 text-[28px] font-medium text-blackMain sm:text-[32px]">
            Welcome Back!
          </p>
          <p className="mb-0 text-[14px] font-normal text-grey">
            Please sign in using your login credentials
          </p>
        </div>
        <div className="px-4 pb-8 pt-2 sm:px-12 sm:pb-16 sm:pt-6">
          <Formik
            initialValues={initialValues}
            validationSchema={LOGIN_SCHEMA}
            onSubmit={handleLogin}
          >
            {() => (
              <Form className="flex flex-col gap-3">
                <TextInput
                  name="email"
                  type="email"
                  label="Email Address"
                  disabled={isLoading}
                />
                <TextInput
                  name="password"
                  type="password"
                  label="Password"
                  placeholder="********"
                  disabled={isLoading}
                />
                <span className="text-xs text-greyLight">
                  Password Hint: At least six (6) digits.
                </span>
                <div className="flex flex-wrap items-center justify-start gap-[10px] pl-0 pt-2 sm:pl-2 sm:pt-5">
                  <Link
                    className="text-sm font-medium text-primary"
                    href="/auth/forgot-password"
                  >
                    Forgot password?
                  </Link>
                </div>
                <div className="mt-10 flex flex-col-reverse justify-between gap-2 sm:flex-row">
                  <Button
                    type="button"
                    onClick={onCreateAccount}
                    variant="lightGreen"
                    className="bg-lightGreenLightest text-sm underline hover:border-lightGreen"
                  >
                    Create Account
                  </Button>

                  <Button variant="primary" type="submit" loading={isLoading}>
                    Log In
                  </Button>
                </div>
              </Form>
            )}
          </Formik>
        </div>
      </div>
    </div>
  );
};

export default Page;
